/*===========================================================================*\
 |    File: CamMap.cpp
 |
 | Purpose: A Modifier and World Space Modifier for camera mapping.
 |
 |          This file contains two plug-ins.  A Object Space Modifier version
 |          and a World Space Modifier (Space Warp) version.
 |          The Modifier code appears after all the Space Warp code.
 |
 | History: Mark Meier, Began 12/15/96.
 |          MM, Modifier added, 12/18/96.
 |          MM, Last Change 12/21/96.
 |          michael malone (mjm) - 2.3.99
 |            made obj & ws mods have similar dlgs
 |            implemented undo/redo in objmod similar to wsmod
 |          michael malone (mjm) - 3.10.99
 |            added multi-channel support
\*===========================================================================*/
/*===========================================================================*\
 | Include Files
\*===========================================================================*/
#include "Max.h"			// Main MAX include file
#include "CamMap.h"			// Resource editor include file
#include "iparamm2.h"


TCHAR *GetString(int id);

/*===========================================================================*\
 | Miscellaneous Defines
\*===========================================================================*/
// The unique ClassIDs

#define MOD_CLASS_ID		Class_ID(0x21ce6eb5, 0x7ed434ec)

// These are the names on the creation buttons
#define OBJ_CLASSNAME		GetString(IDS_CAMMAPCLASSNAME)
#define WSMOD_CLASSNAME		GetString(IDS_CAMERAMAP)
#define CLASSNAME			GetString(IDS_CAMERAMAP)

// These are the categories the buttons go into
#define OBJ_CATEGORY		GetString(IDS_CAMERAMAPPING)
#define WSMOD_CATEGORY		_T("")
#define MOD_CATEGORY		GetString(IDS_MAXSURFACE)

// These are the names that will appear in the Modifier stack
#define OBJ_OBJECT_NAME		GetString(IDS_CAMMAPCLASSNAME)
#define WSMOD_OBJECT_NAME	GetString(IDS_CAMERAMAPBINDING)
#define MOD_OBJECT_NAME		GetString(IDS_CAMERAMAPMODIFIER)

// This is the initial node name for the camera map apparatus object
#define INIT_NODE_NAME		GetString(IDS_CAMMAPCLASSNAME)

// The description that appears in the plug-in summary info dialog box
#define LIBDESCRIPTION		GetString(IDS_CAMMAPTITLE)

// Dialog box message title
#define MESSAGE_TITLE		GetString(IDS_CAMERAMAPPING)


// This is the camera reference index
#define PBLOCK_REF				0

// These are the default colors for the apparatus object in the viewports
#define CAMMAP_R			float(0.7)
#define CAMMAP_G			float(0.0)
#define CAMMAP_B			float(0.0)

// This is the ID of the pick command mode
#define CID_PICK_CAMERA		CID_USER+0x7055

// This is the DLL instance handle
HINSTANCE hInstance;

TCHAR *GetString(int id) {
	static TCHAR buf[256];
	if (hInstance)
		return LoadString(hInstance, id, buf, sizeof(buf)) ? buf : NULL;
	return NULL;
	}




/*===========================================================================*\
 | Modifier Code...
\*===========================================================================*/
/*===========================================================================*\
 | Class definitions
\*===========================================================================*/


class CamMapModValidatorClass : public PBValidator
{
public:
class CamMapMod *mod;
private:
BOOL Validate(PB2Value &v) 
	{
	INode *node = (INode*) v.r;

	if (node->TestForLoop(FOREVER,(ReferenceMaker *) mod)!=REF_SUCCEED) return FALSE;

	if (node) {
		ObjectState os = node->EvalWorldState(0);
		if (os.obj->SuperClassID() == CAMERA_CLASS_ID) {
			return TRUE;
		}
	}

	return FALSE;

	};
};

class CamMapMod : public Modifier { // mjm - 3.10.99
  public:

	CamMapModValidatorClass validator;

	// MAX function interface pointer
	static Interface *ip;

	// Rollup page window handle
	static HWND hModRollup;




// mjm - begin - 2.3.99
	// This is the Modifier being edited...
	static CamMapMod *editMod;

// mjm - end


	// --- Methods From Animatable ---
	void DeleteThis() { delete this; }
	void GetClassName(TSTR& s) { s=GetString(IDS_CAMERAMAPPINGMODIFIER); }
	virtual Class_ID ClassID() { return MOD_CLASS_ID; }
	void BeginEditParams(IObjParam *ip, ULONG flags, Animatable *prev);
	void EndEditParams(IObjParam *ip, ULONG flags, Animatable *next);		

	// --- Methods From ReferenceMaker ---
// mjm - begin - 2.3.99
	RefResult NotifyRefChanged(Interval changeInt, RefTargetHandle hTarget, PartID& partID, RefMessage message);
	int NumRefs() { return 1; } // camera node
	RefTargetHandle GetReference(int i);
	void SetReference(int i, RefTargetHandle rtarg);

	// --- Methods From ReferenceTarget ---
	RefTargetHandle Clone(RemapDir& remap = NoRemap());

	// --- Methods From BaseObject ---
	TCHAR *GetObjectName() { return MOD_OBJECT_NAME; }
	CreateMouseCallBack* GetCreateMouseCallBack() { return NULL; } 

	// --- Methods From Modifier ---
	ChannelMask ChannelsUsed()
		{ return PART_GEOM|PART_TOPO|PART_SELECT|TEXMAP_CHANNEL|PART_VERTCOLOR; } // mjm - 3.10.99
	ChannelMask ChannelsChanged()
		{ return PART_VERTCOLOR|TEXMAP_CHANNEL; } // mjm - 3.10.99
	Class_ID InputType() 
		{ return Class_ID(TRIOBJ_CLASS_ID, 0); }
	void ModifyObject(TimeValue t, ModContext &mc, 
		ObjectState *os, INode *node);
	Interval LocalValidity(TimeValue t);

	// --- Methods From CamMapMod ---
	CamMapMod::CamMapMod();
	CamMapMod::~CamMapMod();
	BOOL StoreCameraData(INode *node);

	IParamBlock2 *pblock2;

// JBW: direct ParamBlock access is added
	int	NumParamBlocks() { return 1; }					// return number of ParamBlocks in this instance
	IParamBlock2* GetParamBlock(int i) { if (i == 0) return pblock2; 
											else return NULL;
												} // return i'th ParamBlock
	IParamBlock2* GetParamBlockByID(BlockID id) {if (pblock2->ID() == id) return pblock2 ;
													else return  NULL; } // return id'd ParamBlock



};

enum { cameramap_params };
enum { cameramap_node, cameramap_channel,cameramap_map };


// Init the class variables (these are shared by each instance of the class).
// This is okay since only one modifier can be edited at a time
Interface *CamMapMod::ip = NULL;
HWND CamMapMod::hModRollup = NULL;
CamMapMod *CamMapMod::editMod = NULL; // mjm - 2.3.99


/*===========================================================================*\
 | Class Descriptor
\*===========================================================================*/
class CamMapModClassDesc : public ClassDesc2 {
  public:
	int 			IsPublic() {return 1;}
	void			*Create(BOOL loading = FALSE) {return new CamMapMod;}
	const TCHAR		*ClassName() {return CLASSNAME;}
	SClass_ID		SuperClassID() {return OSM_CLASS_ID;}
	Class_ID		ClassID() {return MOD_CLASS_ID;}
	const TCHAR		*Category() {return MOD_CATEGORY;}

// JBW: new descriptor data accessors added.  Note that the 
//      internal name is hardwired since it must not be localized.
	const TCHAR*	InternalName() { return _T("CameraMap"); }	// returns fixed parsable name (scripter-visible name)
	HINSTANCE		HInstance() { return hInstance; }			// returns owning module handle

};
static CamMapModClassDesc camMapModDesc;


// per instance geosphere block
static ParamBlockDesc2 cameramap_param_blk ( cameramap_params, _T("Parameters"),  0, &camMapModDesc, P_AUTO_CONSTRUCT + P_AUTO_UI, PBLOCK_REF, 
	//rollout
	IDD_MOD, IDS_PW_PARAMS, 0, 0, NULL,
	// params

	cameramap_node, 	_T("node"),		TYPE_INODE, 		0,				IDS_PW_NODE,
		p_ui, 			TYPE_PICKNODEBUTTON, 	IDC_PICK, 
		end, 
	cameramap_map,  _T("map"), TYPE_INT, 	P_RESET_DEFAULT, 	IDS_PW_MAP, 
		p_default, 		0,	
		p_range, 		0, 1, 
		p_ui, 			TYPE_RADIO,  2,IDC_MAP_CHAN_1,IDC_MAP_CHAN_0,
		end, 

	cameramap_channel,  _T("mapChannel"),	TYPE_INT, 	P_RESET_DEFAULT, 	IDS_PW_CHANNEL, 
		p_default, 		1,	
		p_range, 		1, 99, 
		p_ui, 			TYPE_SPINNER, EDITTYPE_INT, IDC_MAP_CHAN_EDIT,IDC_MAP_CHAN_SPIN,  SPIN_AUTOSCALE,
		end, 


	end
	);


/*===========================================================================*\
 | Dialog Procs
\*===========================================================================*/
//--- SelDlgProc -----------------------------------
 
class CameraDlgProc : public ParamMap2UserDlgProc {
public:
	CamMapMod *mod;		
	CameraDlgProc(CamMapMod *m) {mod = m;}		
	BOOL DlgProc(TimeValue t,IParamMap2 *map,HWND hWnd,UINT msg,WPARAM wParam,LPARAM lParam);		
	void DeleteThis() {delete this;}		
};

BOOL CameraDlgProc::DlgProc (TimeValue t,IParamMap2 *map,HWND hWnd,UINT msg,WPARAM wParam,LPARAM lParam)
	{
	switch (msg) {		
		case WM_COMMAND:
			switch (LOWORD(wParam)) {

				case IDC_HELPBUTTON:
					ShellExecute(GetCOREInterface()->GetMAXHWnd(), "open", "iexplore.exe", "www.max3dstuff.com/max4/cameraMapAnimated/help.html", NULL, SW_SHOWNORMAL);
					break;

		
				}
			break;
		}
	return FALSE;
	}

/*===========================================================================*\
 | CamMapMod MAX Methods
\*===========================================================================*/
// --- Methods From Animatable ---
void CamMapMod::BeginEditParams(IObjParam *ip, ULONG flags, Animatable *prev) {
	this->ip = ip;


// mjm - begin - 2.3.99
	// Indicate this is the current Modifier being edited
	editMod = this;

// mjm - end
	camMapModDesc.BeginEditParams(ip, this, flags, prev);
	cameramap_param_blk.SetUserDlgProc(new CameraDlgProc(this));
	cameramap_param_blk.ParamOption(cameramap_node,p_validator,&validator);

}

void CamMapMod::EndEditParams(IObjParam *ip, ULONG flags, Animatable *next) {

	
	editMod = NULL; // mjm - 2.3.99
	hModRollup = NULL;

	camMapModDesc.EndEditParams(ip, this, flags, next);
}


// --- Methods From Modifier ---


class sMyEnumProc : public DependentEnumProc 
	{
      public :
      virtual int proc(ReferenceMaker *rmaker); 
      INodeTab Nodes;              
	};

int sMyEnumProc::proc(ReferenceMaker *rmaker) 
	{ 
	if (rmaker->SuperClassID()==BASENODE_CLASS_ID)    
			{
            Nodes.Append(1, (INode **)&rmaker);                 
			}
     return 0;              
	}


void CamMapMod::ModifyObject(TimeValue t, ModContext &mc, ObjectState *os, INode *node)
{
	// If the user hasn't choosen a camera yet, don't do anything.
	INode *camRef;
	pblock2->GetValue (cameramap_node,0,camRef,FOREVER);

	if (!camRef) return; // mjm - 2.3.99

	// Make sure this is indeed a TriObject so we can cast it as one...
	assert(os->obj->IsSubClassOf(Class_ID(TRIOBJ_CLASS_ID, 0)));
	TriObject *triObj = (TriObject *)os->obj;
// mjm - begin - 3.10.99
	Mesh &mesh = triObj->GetMesh();
	int nVerts( mesh.getNumVerts() );
	int nFaces( mesh.getNumFaces() );

	// ensure there are tverts for the channel
	int map,channel;
	pblock2->GetValue (cameramap_channel,0,channel,FOREVER);
	pblock2->GetValue (cameramap_map,0,map,FOREVER);
	if (map == 1)
		channel = 0;

	mesh.setMapSupport(channel);

	if ( nVerts != mesh.getNumMapVerts(channel) )
		mesh.setNumMapVerts(channel, nVerts);

	TVFace *tvFaces = mesh.mapFaces(channel);
	for (int i=0; i<nFaces; i++)
		tvFaces[i].setTVerts(mesh.faces[i].getAllVerts());

	// Create a copy of the vertex coordinates and transform them.
	// This first part of the transformation puts the coords into 
	// the space of the camera 

    sMyEnumProc dep;              
	EnumDependents(&dep);

	INode *n0 = dep.Nodes[0];

	// Get the matrix to transform the objects points to world space
	Interval tempIV;
	Matrix3 mat;
	Matrix3 objMat = n0->GetObjectTM(t, &tempIV);
	// Get the matrix to transform the camera into world space
	Matrix3 camMat = camRef->GetObjectTM(t, &tempIV);

	// Create the matrix that represents the relative position of the 
	// object to the camera.
	mat = objMat*Inverse(camMat);

	Point3 *UVW = (Point3 *) LocalAlloc(LPTR, nVerts*sizeof(Point3));
	for (i=0; i<nVerts; i++)
		UVW[i] = mesh.verts[i]*mat;

	// Compute the scale factors
	ObjectState camOState = camRef->EvalWorldState(t);
	float fov = ((CameraObject *)camOState.obj)->GetFOV(t, FOREVER);
	Interface *ip = GetCOREInterface();
	float aspectRatio = ip->GetRendImageAspect();

	float xScale = -0.5f / ((float) tan(0.5*(double)fov));
	float yScale = xScale*aspectRatio;

	// Transform the points into screen space
	float distance, x, y, z;
	for (i=0; i<nVerts; i++) {
		x = UVW[i].x; y = UVW[i].y; z = UVW[i].z;
		distance = (float) sqrt(x*x + y*y + z*z);
		UVW[i].x = UVW[i].x*xScale/z + 0.5f;
		UVW[i].y = UVW[i].y*yScale/z + 0.5f;
		UVW[i].z = distance;
	}

	// We have the UVWs ... set them into the tVerts of the mesh
	UVVert *uvVerts = mesh.mapVerts(channel);
	for (i=0; i<nVerts; i++)
		uvVerts[i] = UVW[i];

	// Free the UVWs we allocated...
	LocalFree(UVW);

	// The texture mapping depends on the geometry and topology so make sure 
	// the validity interval reflects this.
	Interval iv = LocalValidity(t);

	iv &= triObj->ChannelValidity(t, GEOM_CHAN_NUM);
	iv &= triObj->ChannelValidity(t, TOPO_CHAN_NUM);
	iv.Set(t,t);
	if (!channel)
	{
		iv &= triObj->ChannelValidity (t, VERT_COLOR_CHAN_NUM);
		os->obj->UpdateValidity(VERT_COLOR_CHAN_NUM, iv);
	}
	else
	{
		iv &= triObj->ChannelValidity (t, TEXMAP_CHAN_NUM);
		os->obj->UpdateValidity(TEXMAP_CHAN_NUM, iv);
	}
}
// mjm - end

// Returns the validity of the modifier.
Interval CamMapMod::LocalValidity(TimeValue t) {

	INode *camRef;
	pblock2->GetValue (cameramap_node,0,camRef,FOREVER);

	if (camRef) {
		Interval valid = FOREVER;
		Matrix3 tmCam = camRef->GetObjectTM(t, &valid);
		ObjectState camOState = camRef->EvalWorldState(t);
		((CameraObject *) camOState.obj)->GetFOV(t, valid);
		valid.Set(t,t);
		return valid;
	} 
	else
	return FOREVER;
}

// --- Methods From ReferenceMaker ---
RefTargetHandle CamMapMod::GetReference(int i) {
	switch(i) {
		case PBLOCK_REF:  return pblock2;
		default: return NULL;
	}
}

void CamMapMod::SetReference(int i, RefTargetHandle rtarg) { 
	switch(i) {
		case PBLOCK_REF: pblock2 = (IParamBlock2*)rtarg; break;
	}
}

// This method is called when one of the items we reference changes.
RefResult CamMapMod::NotifyRefChanged(Interval changeInt, 
	RefTargetHandle hTarget, PartID& partID, RefMessage message) {
	switch (message) {
		case REFMSG_CHANGE:
			if (hTarget == pblock2)
				{
				ParamID changing_param = pblock2->LastNotifyParamID();

				cameramap_param_blk.InvalidateUI(changing_param);
				}

			break;
	}
	return REF_SUCCEED; 
}

// --- Methods From ReferenceTarget ---
RefTargetHandle CamMapMod::Clone(RemapDir& remap) {
	// Create a new modifier and init the copied vars...
	CamMapMod *newMod = new CamMapMod();

	newMod->ReplaceReference(PBLOCK_REF, pblock2);
	
	BaseClone(this, newMod, remap);
	return newMod;
}

// --- Methods From CamMapMod ---
CamMapMod::CamMapMod() {
	validator.mod = this;
	camMapModDesc.MakeAutoParamBlocks(this);

}

CamMapMod::~CamMapMod() {
	DeleteAllRefsFromMe();	
}

/*

BOOL CamMapMod::StoreCameraData(INode *camNode) {
	ModContextList mcList;
	INodeTab nodeTab;
	Interval valid = FOREVER;
	Matrix3 objMat, camMat;
	TimeValue currentTime = ip->GetTime();

// mjm - begin - 2.3.99
	if (camNode->TestForLoop(FOREVER,this)==REF_SUCCEED) {
		theHold.Begin();

		// Set the camera reference to the node.
		ReplaceReference(CAM_REF, (RefTargetHandle)camNode);

		// Register a restore object with the undo system
		theHold.Put(new PickModCameraRestore(this));
		theHold.Accept(GetString(IDS_PICKCAMERA));
// mjm - end

		// Get the INode* of the _first_ object we modify. Anyone in their 
		// right mind will apply this to only one object, but since it's an
		// OSM they can instance the modifier...
		ip->GetModContexts(mcList, nodeTab);
		INode *n0 = nodeTab[0];

		// Get the matrix to transform the objects points to world space
		objMat = n0->GetObjectTM(currentTime, &valid);
		// Get the matrix to transform the camera into world space
		camMat = camNode->GetObjectTM(currentTime, &valid);

		// Create the matrix that represents the relative position of the 
		// object to the camera.
		mat = objMat*Inverse(camMat);

		// Get the field of view from the camera
		ObjectState camOState = camNode->EvalWorldState(currentTime);
		fov = ((CameraObject *) camOState.obj)->GetFOV(currentTime, valid);

		// Set the flag to indicate we got it...
		haveCameraData = TRUE;

		// Grab the aspect and pixel aspect ratios
		aspectRatio = ip->GetRendImageAspect();

		// We have just changed -- notify our dependents
		NotifyDependents(FOREVER, PART_ALL, REFMSG_CHANGE);
		return TRUE;
// mjm - begin - 2.3.99
	} 
	else {
		// Cyclic reference... cannot assign this node.
		return FALSE;
	}
// mjm - end
}
*/

/*===========================================================================*\
 | Dll/Lib Functions
\*===========================================================================*/
int controlsInit = FALSE;
BOOL WINAPI DllMain(HINSTANCE hinstDLL, ULONG fdwReason, LPVOID lpvReserved) {	
	hInstance = hinstDLL;
	if (! controlsInit) {
		controlsInit = TRUE;
		InitCustomControls(hInstance);
		InitCommonControls();
	}
	return(TRUE);
}

__declspec(dllexport) const TCHAR *LibDescription() {
	return (LIBDESCRIPTION);
}

__declspec(dllexport) int LibNumberClasses() { 
	return 1;
}

__declspec(dllexport) ClassDesc* LibClassDesc(int i) { 
	switch(i) {
		case 0: // Modifier
			return &camMapModDesc; 
		default:
			return 0;
	}
}

__declspec(dllexport) ULONG LibVersion() { 
	return VERSION_3DSMAX; 
}

// Let the plug-in register itself for deferred loading
__declspec( dllexport ) ULONG CanAutoDefer()
{
	return 1;
}

