/**********************************************************************
 *<
	FILE: expgizmo.cpp

	DESCRIPTION: A gizmo helper for the explode atmospheric effect

	CREATED BY: Rolf Berteig

	HISTORY: 4-15-96

 *>	Copyright (c) 1996 Rolf Berteig, All Rights Reserved.
 **********************************************************************/

#include "inferno.h"
#include "iparamm.h"
#include "gizmo.h"
#include "expgizmo.h"

//--- ClassDescriptor and class vars ---------------------------------


class ExpGizmoClassDesc:public ClassDesc {
	public:
	int 			IsPublic() {return 1;}
	void *			Create(BOOL loading = FALSE) {return new ExpGizmoObject;}
	const TCHAR *	ClassName() {return EXPGIZMO_NAME;}
	SClass_ID		SuperClassID() {return HELPER_CLASS_ID;}
	Class_ID		ClassID() {return EXPGIZMO_CLASSID;}
	const TCHAR* 	Category() {return GetString(IDS_RB_ATMOSAPPARATUS);}
	};

static ExpGizmoClassDesc expGizmoDesc;

ClassDesc* GetExpGizmoDesc() {return &expGizmoDesc;}

IParamMap *ExpGizmoObject::pmapParam = NULL;
IObjParam *ExpGizmoObject::ip        = NULL;


//--- Parameter map/block descriptors -------------------------------

static ParamUIDesc descParam[] = {
	// Radius
	ParamUIDesc(
		PB_GIZMO_RADIUS,
		EDITTYPE_UNIVERSE,
		IDC_EXPGIZMO_RADIUS,IDC_EXPGIZMO_RADIUSSPIN,
		0.0f,9999999999.0f,
		SPIN_AUTOSCALE),

	// Hemisphere
	ParamUIDesc(PB_GIZMO_HEMI,TYPE_SINGLECHEKBOX,IDC_EXPGIZMO_HEMI),
	
	// Seed
	ParamUIDesc(
		PB_GIZMO_SEED,
		EDITTYPE_INT,
		IDC_EXPGIZMO_SEED,IDC_EXPGIZMO_SEEDSPIN,
		0.0f,9999999999.0f,
		1.0f),
	};

#define PARAMDESC_LENGH 3

ParamBlockDescID descVer0[] = {
	{ TYPE_FLOAT, NULL, TRUE, 0 }, // Radius
	{ TYPE_INT, NULL, FALSE, 0 },   // Hemi
	{ TYPE_INT, NULL, FALSE, 0 },   // Seed
	};

#define PBLOCK_LENGTH	3
#define CURRENT_DESC	descVer0
#define CURRENT_VERSION	0

//--- ExpGizmoDlgProc --------------------------------

class ExpGizmoDlgProc : public ParamMapUserDlgProc {
	public:
		ExpGizmoObject *ob;

		ExpGizmoDlgProc(ExpGizmoObject *o) {ob=o;}
		BOOL DlgProc(TimeValue t,IParamMap *map,HWND hWnd,UINT msg,WPARAM wParam,LPARAM lParam);
		void DeleteThis() {delete this;}
	};

BOOL ExpGizmoDlgProc::DlgProc(
		TimeValue t,IParamMap *map,HWND hWnd,UINT msg,WPARAM wParam,LPARAM lParam)
	{
	switch (msg) {
		case WM_COMMAND:
			switch (LOWORD(wParam)) {
				case IDC_EXPGIZMO_NEWSEED: {					
					srand(GetTickCount());
					int r = rand();
					ob->pblock->SetValue(PB_GIZMO_SEED,0,r);
					return TRUE;	
					}
				}
			break;	
		}
	return FALSE;
	}


//--- ExpGizmo methods -------------------------------

ExpGizmoObject::ExpGizmoObject()
	{
	MakeRefByID(FOREVER, 0, 
		CreateParameterBlock(
			CURRENT_DESC, 
			PBLOCK_LENGTH, 
			CURRENT_VERSION));
	}

ExpGizmoObject::~ExpGizmoObject()
	{
	}


void ExpGizmoObject::BeginEditParams(
		IObjParam *ip, ULONG flags,Animatable *prev)
	{
	GizmoObject::BeginEditParams(ip,flags,prev);	
	this->ip = ip;

	if (pmapParam) {		
		pmapParam->SetParamBlock(pblock);
	} else {		
		pmapParam = CreateCPParamMap(
			descParam,PARAMDESC_LENGH,
			pblock,
			ip,
			hInstance,
			MAKEINTRESOURCE(IDD_EXPLODE_GIZMO),
			GetString(IDS_RB_COMBUSTPARAMS),
			0);
		}

	pmapParam->SetUserDlgProc(new ExpGizmoDlgProc(this));
	}

void ExpGizmoObject::EndEditParams(
		IObjParam *ip, ULONG flags,Animatable *next)
	{
	GizmoObject::EndEditParams(ip,flags,next);
	this->ip = NULL;

	if (flags&END_EDIT_REMOVEUI ) {		
		DestroyCPParamMap(pmapParam);
		pmapParam  = NULL;		
		}
	}

RefTargetHandle ExpGizmoObject::Clone(RemapDir& remap)
	{
	ExpGizmoObject* newob = new ExpGizmoObject();
	newob->ReplaceReference(0,pblock->Clone(remap));	
	return newob;
	}

void ExpGizmoObject::InvalidateUI()
	{
	if (pmapParam) pmapParam->Invalidate();
	}

ParamDimension *ExpGizmoObject::GetParameterDim(int pbIndex)
	{
	switch (pbIndex) {
		case PB_GIZMO_RADIUS:	return stdWorldDim;
		default:				return defaultDim;
		}
	}

TSTR ExpGizmoObject::GetParameterName(int pbIndex)
	{
	switch (pbIndex) {
		case PB_GIZMO_RADIUS:	return GetString(IDS_RB_CMBRADIUS);
		default:				return GetString(IDS_RB_CMBPARAMETER);			
		}
	}


#define NUM_SEGS	16
static void DrawSphere(
		PolyLineProc& lp, float radius, int hemi)
	{
	float u;
	Point3 pt[3];
	
	// XY
	pt[0] = Point3(radius,0.0f,0.0f);
	for (int i=1; i<=NUM_SEGS; i++) {
		u = float(i)/float(NUM_SEGS) * TWOPI;
		pt[1].x = (float)cos(u) * radius;
		pt[1].y = (float)sin(u) * radius;
		pt[1].z = 0.0f;
		lp.proc(pt,2);
		pt[0] = pt[1];
		}

	// YZ
	pt[0] = Point3(0.0f,radius,0.0f);
	for (i=1; i<=(hemi?NUM_SEGS/2:NUM_SEGS); i++) {
		u = float(i)/float(NUM_SEGS) * TWOPI;
		pt[1].x = 0.0f;
		pt[1].y = (float)cos(u) * radius;
		pt[1].z = (float)sin(u) * radius;
		lp.proc(pt,2);
		pt[0] = pt[1];
		}

	// ZX
	pt[0] = Point3(radius,0.0f,0.0f);
	for (i=1; i<=(hemi?NUM_SEGS/2:NUM_SEGS); i++) {
		u = float(i)/float(NUM_SEGS) * TWOPI;
		pt[1].x = (float)cos(u) * radius;
		pt[1].y = 0.0f;
		pt[1].z = (float)sin(u) * radius;
		lp.proc(pt,2);
		pt[0] = pt[1];
		}
	}

void ExpGizmoObject::DrawGizmo(TimeValue t,GraphicsWindow *gw)
	{
	float radius;
	int hemi;
	pblock->GetValue(PB_GIZMO_RADIUS,t,radius,FOREVER);
	pblock->GetValue(PB_GIZMO_HEMI,t,hemi,FOREVER);
	DrawLineProc proc(gw);
	DrawSphere(proc,radius,hemi);
	}

void ExpGizmoObject::GetBoundBox(
		Matrix3 &mat,TimeValue t,Box3 &box)
	{
	float radius;
	int hemi;
	pblock->GetValue(PB_GIZMO_RADIUS,t,radius,FOREVER);
	pblock->GetValue(PB_GIZMO_HEMI,t,hemi,FOREVER);
	BoxLineProc proc(&mat);
	DrawSphere(proc,radius,hemi);
	box += proc.Box();
	}


class ExpGizmoCreateCallBack: public CreateMouseCallBack {
	ExpGizmoObject *ob;
	Point3 p0,p1;
	IPoint2 sp0, sp1;
	
	public:
		int proc( ViewExp *vpt,int msg, int point, int flags, IPoint2 m, Matrix3& mat );
		void SetObj(ExpGizmoObject *obj) {ob = obj;}
	};

int ExpGizmoCreateCallBack::proc(
		ViewExp *vpt,int msg, int point, int flags, 
		IPoint2 m, Matrix3& mat) 
	{	
	if (msg==MOUSE_POINT||msg==MOUSE_MOVE) {
		switch(point) {
			case 0:
				sp0 = m;
				ob->pblock->SetValue(PB_GIZMO_RADIUS,0,0.0f);
				p0 = vpt->SnapPoint(m,m,NULL,SNAP_IN_PLANE);
				p1 = p0 + Point3(.01,.01,.01);
				mat.SetTrans(p0);
				break;
			case 1:
				sp1 = m;
				p1 = vpt->SnapPoint(m,m,NULL,SNAP_IN_PLANE);
				mat.SetTrans(p0);
				ob->pblock->SetValue(PB_GIZMO_RADIUS,0,
					Length(p1-p0));
				ob->pmapParam->Invalidate();										

				if (msg==MOUSE_POINT) {				
					return (Length(sp1-sp0)<3)?CREATE_ABORT:CREATE_STOP;
					}					
				break;
			}
		}
	else
	if (msg == MOUSE_ABORT) {		
		return CREATE_ABORT;
		}

	return TRUE;
	}

static ExpGizmoCreateCallBack expGizmoCreateCB;

CreateMouseCallBack* ExpGizmoObject::GetCreateMouseCallBack()
	{
	expGizmoCreateCB.SetObj(this);
	return &expGizmoCreateCB;
	}
