/**********************************************************************
 *<
	FILE: explode.cpp

	DESCRIPTION: A fire/explosion object atmospheric effect

	CREATED BY: Peter Watje	

	HISTORY: 12-30-96

 *>	Copyright (c) 1996 Peter Watje, All Rights Reserved.
 **********************************************************************/

#include "inferno.h"
#include "imtl.h"
#include "render.h"  
#include <bmmlib.h>
#include "iparamm.h"
#include "texutil.h"
//#include "gizmo.h"
//#include "expgizmo.h"
#include "istdplug.h"
#include "Simpobj.h"



static Class_ID volumeExplode(0x104d4ed3, 0xb43789c);

#define EXPLODE_CLASSNAME GetString(IDS_RB_COMBUSTION)

#define PBLOCK_REF	0
#define OBJECT_REF	1

#define A_RENDER			A_PLUGIN1


class NullView: public View {
	public:
		Point2 ViewToScreen(Point3 p) { return Point2(p.x,p.y); }
		NullView() { worldToView.IdentityMatrix(); screenW=640.0f; screenH = 480.0f; }
	};


class ExplodeDlgProc;



class ExplodeSource {
	public:
		Matrix3 tm;
		int hemi;
		Point3 seedPt;
		GeomObject *obj;
		float radius, radius2, hemiRange, hemiLen;
		ExplodeSource(INode *node,
			float drift, float rfact,
			TimeValue t, Interval &valid, int rseed);
		ExplodeSource();
	};

class ExplodeAtmos : public Atmospheric {
	public:
		// Parameters
		IParamBlock *pblock;  // Ref #0
		Tab<INode*> nodes;	  // Ref #1-n

		// Caches
		Tab<ExplodeSource> sources;		
		float stretch, regularity, drift, rfact;
		float density, scale, phase, levels, fury;
		float object_size,trace_back;
		float falloff;
		int density_path,size_path;
		int localize_explosion;
		int localize_noise;
		int rseed;
		Color color1, color2, color3;
		int samples, invert, explode, smoke;


		TimeValue DataTime;

		Interval valid;		
		CRITICAL_SECTION csect;

		static ExplodeDlgProc *dlg;

		ExplodeAtmos();
		~ExplodeAtmos() {DeleteCriticalSection(&csect);}

		void UpdateCaches(TimeValue t);

		// Animatable/Reference
		int NumSubs() {return 1;}
		Animatable* SubAnim(int i) {return pblock;}
		TSTR SubAnimName(int i) {return _T("");}
		int NumRefs();
		RefTargetHandle GetReference(int i);
		void SetReference(int i, RefTargetHandle rtarg);
		Class_ID ClassID() {return volumeExplode;}
		void GetClassName(TSTR& s) {s=EXPLODE_CLASSNAME;}
		void DeleteThis() {delete this;}
		RefResult NotifyRefChanged(Interval changeInt, RefTargetHandle hTarget, 
	         PartID& partID,  RefMessage message);
		IOResult Load(ILoad *iload);
		IOResult Save(ISave *isave);

		// Atmospheric
		TSTR GetName() {return EXPLODE_CLASSNAME;}
		AtmosParamDlg *CreateParamDialog(IRendParams *ip);
		int RenderBegin(TimeValue t, ULONG flags);
		int RenderEnd(TimeValue t);
		void Update(TimeValue t, Interval& valid);
		void Shade(ShadeContext& sc,const Point3& p0,const Point3& p1,Color& color, Color& trans, BOOL isBG);

		void TraceExplosion(
			ExplodeSource &src, Ray ray, Ray oray, float len,
			Color &c, float &o,
			TimeValue t);
	};

class ExplodeParamDlg : public AtmosParamDlg {
	public:
		ExplodeAtmos *atmos;
		IRendParams *ip;
		IParamMap *pmap;

		ExplodeParamDlg(ExplodeAtmos *a,IRendParams *i);
		Class_ID ClassID() {return volumeExplode;}
		ReferenceTarget* GetThing() {return atmos;}
		void SetThing(ReferenceTarget *m);		
		void DeleteThis();
	};


class ExplodeClassDesc:public ClassDesc {
	public:
	int 			IsPublic() { return 1; }
	void *			Create(BOOL loading) { return new ExplodeAtmos; }
	const TCHAR *	ClassName() { return EXPLODE_CLASSNAME; }
	SClass_ID		SuperClassID() { return ATMOSPHERIC_CLASS_ID; }
	Class_ID 		ClassID() { return volumeExplode; }
	const TCHAR* 	Category() { return _T("");  }
	};

static ExplodeClassDesc explodeCD;
ClassDesc* GetExplodeDesc() {return &explodeCD;}

#define PB_COLOR1			0
#define PB_COLOR2			1
#define PB_COLOR3			2
#define PB_FLAMETYPE		3
#define PB_STRETCH			4
#define PB_REGULARITY		5
#define PB_SCALE			6
#define PB_LEVELS			7
#define PB_DENSITY			8
#define PB_SAMPLES			9
#define PB_PHASE			10
#define PB_DRIFT			11
#define PB_EXPLODE			12
#define PB_SMOKE			13
#define PB_FURY				14
#define PB_OBJECT_SIZE		15
#define PB_TRACE_BACK		16
#define PB_DENSITY_PATH		17
#define PB_SIZE_PATH		18
#define PB_LOCALIZE_EXPLOSION 19
#define PB_LOCALIZE_NOISE	20
#define PB_SEED				21
#define PB_FALLOFF			22

static int typeIDs[] = {IDC_EXPLODE_FIREBALL,IDC_EXPLODE_TENDRIL};

static ParamUIDesc descParam[] = {
	
	// Color 1
	ParamUIDesc(PB_COLOR1,TYPE_COLORSWATCH,IDC_EXPLODE_COLOR1),

	// Color 2
	ParamUIDesc(PB_COLOR2,TYPE_COLORSWATCH,IDC_EXPLODE_COLOR2),

	// Color 3
	ParamUIDesc(PB_COLOR3,TYPE_COLORSWATCH,IDC_EXPLODE_COLOR3),

	// Type
	ParamUIDesc(PB_FLAMETYPE,TYPE_RADIO,typeIDs,2),

	// Stretch
	ParamUIDesc(
		PB_STRETCH,
		EDITTYPE_FLOAT,
		IDC_EXPLODE_STRETCH,IDC_EXPLODE_STRETCHSPIN,
		0.0f,999999999.0f,
		0.1f),	
	
	// Regularity
	ParamUIDesc(
		PB_REGULARITY,
		EDITTYPE_FLOAT,
		IDC_EXPLODE_REG,IDC_EXPLODE_REGSPIN,
		0.0f,1.0f,
		0.01f),
		
	// Scale
	ParamUIDesc(
		PB_SCALE,
		EDITTYPE_FLOAT,
		IDC_EXPLODE_SCALE,IDC_EXPLODE_SCALESPIN,
		0.0f,999999999.0f,
		0.01f),
	
	// Levels
	ParamUIDesc(
		PB_LEVELS,
		EDITTYPE_FLOAT,
		IDC_EXPLODE_LEVELS,IDC_EXPLODE_LEVELSSPIN,
		1.0f,10.0f,
		0.01f),

	// Density
	ParamUIDesc(
		PB_DENSITY,
		EDITTYPE_FLOAT,
		IDC_EXPLODE_DENISITY,IDC_EXPLODE_DENISITYSPIN,
		0.0f,999999999.0f,
		0.1f),	

	// Samples
	ParamUIDesc(
		PB_SAMPLES,
		EDITTYPE_INT,
		IDC_EXPLODE_SAMPLES,IDC_EXPLODE_SAMPLESSPIN,
		1.0f,100.0f,
		0.5f),		

	// Phase
	ParamUIDesc(
		PB_PHASE,
		EDITTYPE_FLOAT,
		IDC_EXPLODE_PHASE,IDC_EXPLODE_PHASESPIN,
		0.0f,999999999.0f,
		0.01f),
	
	// Drift
	ParamUIDesc(
		PB_DRIFT,
		EDITTYPE_FLOAT,
		IDC_EXPLODE_DRIFT,IDC_EXPLODE_DRIFTSPIN,
		0.0f,999999999.0f,
		0.01f),

	// Explode
	ParamUIDesc(PB_EXPLODE,TYPE_SINGLECHEKBOX,IDC_EXPLODE_ON),

	// Smoke
	ParamUIDesc(PB_SMOKE,TYPE_SINGLECHEKBOX,IDC_EXPLODE_SMOKE),
	
	// Fury
	ParamUIDesc(
		PB_FURY,
		EDITTYPE_FLOAT,
		IDC_EXPLODE_FURY,IDC_EXPLODE_FURYSPIN,
		0.0f,999999999.0f,
		0.01f),	

	// Object Size
	ParamUIDesc(
		PB_OBJECT_SIZE,
		EDITTYPE_FLOAT,
		IDC_OBJECT_SIZE,IDC_OBJECT_SIZESPIN,
		0.0f,999999.0f,
		1.0f),

	// Trace BAck
	ParamUIDesc(
		PB_TRACE_BACK,
		EDITTYPE_FLOAT,
		IDC_TRACE_BACK,IDC_TRACE_BACKSPIN,
		0.0f,999999.0f,
		1.0f),

	// Denisty Path
	ParamUIDesc(PB_DENSITY_PATH,TYPE_SINGLECHEKBOX,IDC_DPATH_CHECK),

	// Life Path
	ParamUIDesc(PB_SIZE_PATH,TYPE_SINGLECHEKBOX,IDC_SPATH_CHECK),

	// Localize explosion
	ParamUIDesc(PB_LOCALIZE_EXPLOSION,TYPE_SINGLECHEKBOX,IDC_LOCALIZE_EXPLOSION_CHECK),

	// Localize noise
	ParamUIDesc(PB_LOCALIZE_NOISE,TYPE_SINGLECHEKBOX,IDC_LOCALIZE_NOISE_CHECK),


	// seed
	ParamUIDesc(
		PB_SEED,
		EDITTYPE_INT,
		IDC_SEED,IDC_SEED_SPIN,
		0.0f,999999.0f,
		1.0f),

	// falloff
	ParamUIDesc(
		PB_FALLOFF,
		EDITTYPE_FLOAT,
		IDC_FALLOFF,IDC_FALLOFF_SPIN,
		0.1f,1.0f,
		0.1f),


	};

#define PARAMDESC_LENGH 23

static ParamBlockDescID descVer1[] = {
	{ TYPE_POINT3, NULL, TRUE, 0 }, // color 1
	{ TYPE_POINT3, NULL, TRUE, 1 }, // color 2
	{ TYPE_POINT3, NULL, TRUE, 2 }, // color 3
	{ TYPE_INT, NULL, FALSE, 3 },	// type
	{ TYPE_FLOAT, NULL, TRUE, 4 },	// stretch
	{ TYPE_FLOAT, NULL, TRUE, 5 },	// regularity
	{ TYPE_FLOAT, NULL, TRUE, 6 },	// scale
	{ TYPE_FLOAT, NULL, TRUE, 7 },	// levels
	{ TYPE_FLOAT, NULL, TRUE, 8 },	// density	
	{ TYPE_INT, NULL, FALSE, 9 },	// samples	
	{ TYPE_FLOAT, NULL, TRUE, 10 },	// phase
	{ TYPE_FLOAT, NULL, TRUE, 11 },	// drift
	{ TYPE_INT, NULL, FALSE, 12 },	// explode
	{ TYPE_INT, NULL, FALSE, 13 },// smoke
	{ TYPE_FLOAT, NULL, FALSE, 14 },// fury
	{ TYPE_FLOAT, NULL, TRUE, 15 },// particle size
	{ TYPE_INT, NULL, TRUE, 16 },// particle life
	{ TYPE_INT, NULL, FALSE, 17 },// density path
	{ TYPE_INT, NULL, FALSE, 18 },// size path
	{ TYPE_INT, NULL, FALSE, 19 },// localize explosion
	{ TYPE_INT, NULL, FALSE, 20 },// localize noise
	{ TYPE_INT, NULL, FALSE, 21 },// seed
	{ TYPE_FLOAT, NULL, TRUE, 22 }// FALLOFF
	};

#define CURRENT_DESCRIPTOR descVer1

#define PBLOCK_LENGTH	23

#define CURRENT_VERSION	1


//--- ExplodeDlgProc ----------------------------------------------------------

class ExplodeDlgProc : 
			public ParamMapUserDlgProc,
			public RendPickProc {
	public:
		ExplodeAtmos *atmos;
		IRendParams *ip;
		ICustButton *iPick, *iRemove;		
		ISpinnerControl *iFury;
		HWND hWnd;
		HFONT hFont;

		ExplodeDlgProc(ExplodeAtmos *v,IRendParams *i);
		
		void Init(HWND hWnd);		
		BOOL DlgProc(TimeValue t,IParamMap *map,HWND hWnd,UINT msg,WPARAM wParam,LPARAM lParam);
		void DeleteThis();
		void UpdateNames();		
		void SetStates(HWND hWnd);
		void SetupExplosion(HWND hWnd);
		void DoAbout(HWND hWnd);

		// From RendPickProc
		BOOL Pick(INode *node);
		void Remove();
		BOOL Filter(INode *node);
		void EnterMode() {iPick->SetCheck(TRUE);}
		void ExitMode() {iPick->SetCheck(FALSE);}
	};

ExplodeDlgProc::ExplodeDlgProc(ExplodeAtmos *v,IRendParams *i) 
	{
	atmos   = v;
	ip      = i;
	iPick   = NULL;
	iFury   = NULL;
	iRemove = NULL;	
	atmos->dlg=this;
	}

void ExplodeDlgProc::DeleteThis() 
	{	
	atmos->dlg = NULL;
	ip->EndPickMode();
	ReleaseICustButton(iPick);
	ReleaseICustButton(iRemove);	
	ReleaseISpinner(iFury);
	DeleteObject(hFont);
	delete this;
	}

BOOL ExplodeDlgProc::Pick(INode *node)
	{
	if (atmos->nodes.Count() && atmos->nodes[0]==NULL) {
		atmos->ReplaceReference(OBJECT_REF,node);
	} else {
		atmos->nodes.SetCount(atmos->nodes.Count()+1);
		atmos->nodes[atmos->nodes.Count()-1] = NULL;
		atmos->ReplaceReference(OBJECT_REF+atmos->nodes.Count()-1,node);
		}
	UpdateNames();
	return TRUE;
	}

void ExplodeDlgProc::Remove()
	{
	int sel = SendMessage(GetDlgItem(hWnd,IDC_EXPLODE_OBJECTNAME),CB_GETCURSEL,0,0);
	if (sel!=CB_ERR) {
		int c=0;
		for (int i=0; i<atmos->nodes.Count(); i++) {
			if (atmos->nodes[i]) {
				if (sel==c) {
					atmos->DeleteReference(i+OBJECT_REF);
					atmos->nodes.Delete(i,1);
					UpdateNames();
					break;
					}
				c++;
				}
			}
		}
	}

BOOL ExplodeDlgProc::Filter(INode *node)
	{
	for (int i=0; i<atmos->nodes.Count(); i++) {
		if (atmos->nodes[i]==node) return FALSE;
		}
	ObjectState os = node->EvalWorldState(ip->GetTime());
	if (os.obj->SuperClassID()==GEOMOBJECT_CLASS_ID &&
		os.obj->IsRenderable()) return TRUE;
	else return FALSE;

//filter only particle objects
//	if (os.obj->GetInterface(I_PARTICLEOBJ)==NULL)
//		return FALSE;
//		else return TRUE;
//	return os.obj->ClassID()==EXPGIZMO_CLASSID;
	}

void ExplodeDlgProc::UpdateNames()
	{
	int c=0;
	SendMessage(GetDlgItem(hWnd,IDC_EXPLODE_OBJECTNAME),CB_RESETCONTENT,0,0);
	for (int i=0; i<atmos->nodes.Count(); i++) {
		if (atmos->nodes[i]) {
			c++;
			SendMessage(GetDlgItem(hWnd,IDC_EXPLODE_OBJECTNAME),
				CB_ADDSTRING,0,(LPARAM)(const TCHAR*)atmos->nodes[i]->GetName());
			}
		}
	SendMessage(GetDlgItem(hWnd,IDC_EXPLODE_OBJECTNAME),CB_SETCURSEL,0,0);
	if (c) {
		iRemove->Enable();
	} else {
		iRemove->Disable();
		}
	}

void ExplodeDlgProc::SetStates(HWND hWnd)
	{
	int explode;
	atmos->pblock->GetValue(PB_EXPLODE,0,explode,FOREVER);
	if (explode) {		
		iFury->Enable();
		EnableWindow(GetDlgItem(hWnd,IDC_EXPLODE_SMOKE),TRUE);
		EnableWindow(GetDlgItem(hWnd,IDC_EXPLODE_SETUPPHASE),TRUE);		
		EnableWindow(GetDlgItem(hWnd,IDC_EXPLODE_FURYLABEL),TRUE);		
	} else {		
		iFury->Disable();
		EnableWindow(GetDlgItem(hWnd,IDC_EXPLODE_SMOKE),FALSE);
		EnableWindow(GetDlgItem(hWnd,IDC_EXPLODE_SETUPPHASE),FALSE);		
		EnableWindow(GetDlgItem(hWnd,IDC_EXPLODE_FURYLABEL),FALSE);
		}
	}


void ExplodeDlgProc::Init(HWND hWnd)
	{
	this->hWnd = hWnd;
	iPick = GetICustButton(GetDlgItem(hWnd,IDC_EXPLODE_PICK));
	iPick->SetType(CBT_CHECK);
	iPick->SetHighlightColor(GREEN_WASH);
	iRemove = GetICustButton(GetDlgItem(hWnd,IDC_EXPLODE_REMOVE));	
	iFury   = GetISpinner(GetDlgItem(hWnd,IDC_EXPLODE_FURYSPIN));
	hFont   =  CreateFont(24,0,0,0,FW_BOLD,0,0,0,0,0,0,0, VARIABLE_PITCH | FF_SWISS, _T(""));
	SendDlgItemMessage(hWnd,IDC_INFERNO_TITLE,WM_SETFONT,(WPARAM)hFont,TRUE);
	UpdateNames();
	}


BOOL ExplodeDlgProc::DlgProc(
		TimeValue t,IParamMap *map,HWND hWnd,
		UINT msg,WPARAM wParam,LPARAM lParam)
	{
	switch (msg) {
		case WM_INITDIALOG:
			Init(hWnd);
			SetStates(hWnd);
			break;

		case WM_COMMAND:
			switch (LOWORD(wParam)) {
				case IDC_EXPLODE_PICK:
					ip->SetPickMode(this);
					break;

				case IDC_EXPLODE_REMOVE:
					Remove();
					break;

				case IDC_EXPLODE_SETUPPHASE:
					SetupExplosion(hWnd);
					break;

				case IDC_EXPLODE_ON:
					SetStates(hWnd);
					break;

				case IDC_EXPLODE_ABOUT:
					DoAbout(hWnd);
					break;
				}
			break;
		}
	
	return FALSE;
	}

static BOOL CALLBACK SetupExplodeDlgProc(
		HWND hWnd, UINT msg, WPARAM wParam, LPARAM lParam)
	{
	static Interval *range = NULL;

	switch (msg) {
		case WM_INITDIALOG: {
			range = (Interval*)lParam;
			ISpinnerControl *spin;
			spin = GetISpinner(GetDlgItem(hWnd,IDC_EXPLODE_STARTSPIN));
			spin->LinkToEdit(GetDlgItem(hWnd,IDC_EXPLODE_START),EDITTYPE_TIME);
			spin->SetLimits(TIME_NegInfinity,TIME_PosInfinity, FALSE);
			spin->SetValue(range->Start(),FALSE);
			spin->SetScale(10.0f);
			ReleaseISpinner(spin);

			spin = GetISpinner(GetDlgItem(hWnd,IDC_EXPLODE_ENDSPIN));
			spin->LinkToEdit(GetDlgItem(hWnd,IDC_EXPLODE_END),EDITTYPE_TIME);
			spin->SetLimits(TIME_NegInfinity,TIME_PosInfinity, FALSE);
			spin->SetValue(range->End(),FALSE);
			spin->SetScale(10.0f);
			ReleaseISpinner(spin);

			CenterWindow(hWnd, GetParent(hWnd));
			break;
			}

		case WM_COMMAND:
			switch (LOWORD(wParam)) {
				case IDOK: {
					ISpinnerControl *spin;
					TimeValue start, end;
					spin  = GetISpinner(GetDlgItem(hWnd,IDC_EXPLODE_STARTSPIN));
					start = spin->GetIVal();
					ReleaseISpinner(spin);
					spin  = GetISpinner(GetDlgItem(hWnd,IDC_EXPLODE_ENDSPIN));
					end   = spin->GetIVal();
					ReleaseISpinner(spin);
					range->Set(start,end);
					EndDialog(hWnd,1);
					break;
					}

				case IDCANCEL:
					EndDialog(hWnd,0);
					break;
				}
			break;

		default:
			return FALSE;
		}
	return TRUE;
	}

void ExplodeDlgProc::SetupExplosion(HWND hWnd)
	{
	static Interval iv(0,16000);
	if (!DialogBoxParam(
		hInstance,
		MAKEINTRESOURCE(IDD_EXPLODE_SETUP),
		hWnd,
		SetupExplodeDlgProc,
		(LONG)&iv)) return;	
	
	// Create a fresh bezier float controller.
	Control *cont = (Control*)CreateInstance(
		CTRL_FLOAT_CLASS_ID,
		Class_ID(HYBRIDINTERP_FLOAT_CLASS_ID,0));
	if (!cont) return;
	IKeyControl *ikey = GetKeyControlInterface(cont);
	if (!ikey) {
		cont->DeleteThis();
		return;
		}
	
	// Setup a key structure with slow in and fast out tangents
	IBezFloatKey key;
	SetInTanType(key.flags,BEZKEY_SLOW);
	SetOutTanType(key.flags,BEZKEY_FAST);
	key.intan = key.outtan = 0.0f;

	// Make the first key
	key.val  = 0.0f;
	key.time = iv.Start();
	ikey->AppendKey(&key);

	// Make the second key
	key.val  = 300.0f;
	key.time = iv.End();
	ikey->AppendKey(&key);

	// Sort the table
	ikey->SortKeys();

	// Replace the controller in the param block with this controller.
	atmos->pblock->
		SetController(PB_PHASE,cont,FALSE);
	}

static BOOL CALLBACK AboutDlgProc(
		HWND hWnd, UINT msg, WPARAM wParam, LPARAM lParam)
	{	
	switch (msg) {
		case WM_INITDIALOG:			
			SendDlgItemMessage(hWnd,IDC_INFERNO_TITLE,WM_SETFONT,(WPARAM)lParam,TRUE);
			CenterWindow(hWnd, GetParent(hWnd));
			break;
			
		case WM_COMMAND:
			switch (LOWORD(wParam)) {
				case IDOK:
					EndDialog(hWnd,0);
					break;
				}
			break;

		default:
			return FALSE;
		}
	return TRUE;
	}

void ExplodeDlgProc::DoAbout(HWND hWnd)
	{
	DialogBoxParam(
		hInstance,
		MAKEINTRESOURCE(IDD_EXPLODE_ABOUT),
		hWnd,
		AboutDlgProc,
		(LPARAM)hFont);	
	}


//--- ExplodeParamDlg -------------------------------------------------------

ExplodeParamDlg::ExplodeParamDlg(ExplodeAtmos *a,IRendParams *i) 
	{
	atmos = a;
	ip    = i;	
	pmap  = CreateRParamMap(
		descParam,PARAMDESC_LENGH,
		atmos->pblock,
		i,
		hInstance,
		MAKEINTRESOURCE(IDD_EXPLODE_PARAMS),
		GetString(IDS_RB_COMBUSTPARAMS),
		0);	
	
	pmap->SetUserDlgProc(new ExplodeDlgProc(atmos,ip));
	}

void ExplodeParamDlg::SetThing(ReferenceTarget *m)
	{
	assert(m->ClassID()==atmos->ClassID());
	atmos = (ExplodeAtmos*)m;
	pmap->SetParamBlock(atmos->pblock);	
	if (atmos->dlg) {
		atmos->dlg->atmos = atmos;
		atmos->dlg->UpdateNames();
		atmos->dlg->SetStates(atmos->dlg->hWnd);
		}
	}

void ExplodeParamDlg::DeleteThis()
	{
	DestroyRParamMap(pmap);
	delete this;
	}



//--- ExplodeAtmos -------------------------------------------------------

ExplodeDlgProc *ExplodeAtmos::dlg = NULL;

#define CF(c) (float(c)/float(255))

ExplodeAtmos::ExplodeAtmos()
	{
	InitializeCriticalSection(&csect);
	MakeRefByID(FOREVER, 0, 
		CreateParameterBlock(
			CURRENT_DESCRIPTOR, 
			PBLOCK_LENGTH, 
			CURRENT_VERSION));
	assert(pblock);		

	pblock->SetValue(PB_COLOR1,0,Point3(CF(252),CF(202),CF(  0)));
	pblock->SetValue(PB_COLOR2,0,Point3(CF(225),CF( 30),CF( 30)));	
	pblock->SetValue(PB_COLOR3,0,Point3(0.1,0.1,0.1));
	pblock->SetValue(PB_FLAMETYPE,0,0);
	pblock->SetValue(PB_STRETCH,0,1.0f);
	pblock->SetValue(PB_REGULARITY,0,0.2f);
	pblock->SetValue(PB_SCALE,0,35.0f);	
	pblock->SetValue(PB_LEVELS,0,3.0f);
	pblock->SetValue(PB_DENSITY,0,15.0f);	
	pblock->SetValue(PB_SAMPLES,0,15);			
	pblock->SetValue(PB_PHASE,0,0.0f);
	pblock->SetValue(PB_DRIFT,0,0.0f);
	pblock->SetValue(PB_EXPLODE,0,0);
	pblock->SetValue(PB_SMOKE,0,1);
	pblock->SetValue(PB_FURY,0,1.0f);
	pblock->SetValue(PB_OBJECT_SIZE,0,100.0f);
	pblock->SetValue(PB_TRACE_BACK,0,500.0f);
	pblock->SetValue(PB_DENSITY_PATH,0,0);
	pblock->SetValue(PB_SIZE_PATH,0,0);
	pblock->SetValue(PB_LOCALIZE_EXPLOSION,0,0);
	pblock->SetValue(PB_LOCALIZE_NOISE,0,0);
	pblock->SetValue(PB_SEED,0,0);
	pblock->SetValue(PB_FALLOFF,0,1.0f);
	DataTime = -1;
	valid.SetEmpty();
	}

#define NUMOBJECTS_CHUNK	0x0010

IOResult ExplodeAtmos::Load(ILoad *iload)
	{

	Atmospheric::Load(iload);
	
	ULONG nb;
	int num;
	IOResult res = IO_OK;
	while (IO_OK==(res=iload->OpenChunk())) {
		switch (iload->CurChunkID()) {
			case NUMOBJECTS_CHUNK: {
				res = iload->Read(&num,sizeof(num),&nb);
				nodes.SetCount(num);
				for (int i=0; i<num; i++) nodes[i] = NULL;
				break;
				}
			}
		iload->CloseChunk();
		if (res!=IO_OK)  return res;
		}	
	
	return IO_OK;
	}

IOResult ExplodeAtmos::Save(ISave *isave)
	{

	Atmospheric::Save(isave);
	
	ULONG nb;
	int num = nodes.Count();
		
	isave->BeginChunk(NUMOBJECTS_CHUNK);
	isave->Write(&num,sizeof(num),&nb);
	isave->EndChunk();	
	
	return IO_OK;
	}

int ExplodeAtmos::NumRefs() 
	{
	return 1+nodes.Count();
	}

RefTargetHandle ExplodeAtmos::GetReference(int i) 
	{
	switch (i) {
		case PBLOCK_REF:	return pblock;		
		default:
			if (i>=OBJECT_REF) return nodes[i-OBJECT_REF];
			else return NULL;			
		}
	}

void ExplodeAtmos::SetReference(int i, RefTargetHandle rtarg) 
	{	
	switch (i) {
		case PBLOCK_REF:	pblock = (IParamBlock*)rtarg; break;
		default:
			if (i>=OBJECT_REF) 
				nodes[i-OBJECT_REF] = (INode*)rtarg; 
			break;
		}
	}
			
RefResult ExplodeAtmos::NotifyRefChanged(
		Interval changeInt, RefTargetHandle hTarget,
		PartID& partID,  RefMessage message) 
	{
	switch (message) {
		case REFMSG_TARGET_DELETED: {
			for (int i=0; i<nodes.Count(); i++) {
				if (hTarget==nodes[i]) {
					nodes[i] = NULL;					
					}
				}
			if (dlg) dlg->UpdateNames();
			break;
			}

		case REFMSG_CHANGE:
			valid.SetEmpty();
			break;

		case REFMSG_NODE_NAMECHANGE:
			if (dlg) dlg->UpdateNames();
			break;

		case REFMSG_GET_PARAM_DIM: {
			GetParamDim *gpd = (GetParamDim*)partID;
			switch (gpd->index) {			
				case PB_COLOR1: 	gpd->dim = stdColor255Dim; break;
				case PB_COLOR2: 	gpd->dim = stdColor255Dim; break;
				case PB_COLOR3: 	gpd->dim = stdColor255Dim; break;
				default: 			gpd->dim = defaultDim;
				}
			return REF_STOP; 
			}

		case REFMSG_GET_PARAM_NAME: {
			GetParamName *gpn = (GetParamName*)partID;
			switch (gpn->index) {				
				case PB_COLOR1:		gpn->name = GetString(IDS_RB_INNERCOLOR); break;
				case PB_COLOR2:		gpn->name = GetString(IDS_RB_OUTERCOLOR); break;
				case PB_COLOR3:		gpn->name = GetString(IDS_RB_SMOKECOLOR); break;
				case PB_STRETCH:	gpn->name = GetString(IDS_RB_STRETCH); break;
				case PB_REGULARITY:	gpn->name = GetString(IDS_RB_REGULARITY); break;
				case PB_SCALE:		gpn->name = GetString(IDS_RB_FLAMESIZE); break;
				case PB_LEVELS:		gpn->name = GetString(IDS_RB_FLAMEDETAIL); break;
				case PB_DENSITY:	gpn->name = GetString(IDS_RB_DENSITY); break;				
				case PB_PHASE:		gpn->name = GetString(IDS_RB_PHASE); break;
				case PB_DRIFT:		gpn->name = GetString(IDS_RB_DRIFT); break;
				case PB_OBJECT_SIZE:	gpn->name = GetString(IDS_RB_PARTICLE_SIZE); break;
				case PB_TRACE_BACK:	gpn->name = GetString(IDS_RB_PARTICLE_LIFE); break;
				case PB_FALLOFF:	gpn->name = GetString(IDS_RB_FALLOFF); break;
				default:			gpn->name = _T("?????"); break;
				}
			return REF_STOP; 
			}
		}
	return REF_SUCCEED;
	}

AtmosParamDlg *ExplodeAtmos::CreateParamDialog(IRendParams *ip)
	{
	return new ExplodeParamDlg(this,ip);
	}

void ExplodeAtmos::UpdateCaches(TimeValue t)
	{		
//ebugPrint("Updating Caches\n");
	EnterCriticalSection(&csect);
	if (!valid.InInterval(t)) {
		valid = FOREVER;				
		pblock->GetValue(PB_COLOR1,t,color1,valid);
		pblock->GetValue(PB_COLOR2,t,color2,valid);
		pblock->GetValue(PB_COLOR3,t,color3,valid);
		pblock->GetValue(PB_FLAMETYPE,t,invert,valid);
		pblock->GetValue(PB_STRETCH,t,stretch,valid);
		pblock->GetValue(PB_REGULARITY,t,regularity,valid);
		pblock->GetValue(PB_SCALE,t,scale,valid);
		pblock->GetValue(PB_LEVELS,t,levels,valid);
		pblock->GetValue(PB_DENSITY,t,density,valid);
		pblock->GetValue(PB_SAMPLES,t,samples,valid);		
		pblock->GetValue(PB_PHASE,t,phase,valid);
		pblock->GetValue(PB_DRIFT,t,drift,valid);
		pblock->GetValue(PB_EXPLODE,t,explode,valid);
		pblock->GetValue(PB_SMOKE,t,smoke,valid);
		pblock->GetValue(PB_FURY,t,fury,valid);		
		pblock->GetValue(PB_OBJECT_SIZE,t,object_size,valid);		
		pblock->GetValue(PB_TRACE_BACK,t,trace_back,valid);		
		pblock->GetValue(PB_DENSITY_PATH,t,density_path,valid);
		pblock->GetValue(PB_SIZE_PATH,t,size_path,valid);
		pblock->GetValue(PB_LOCALIZE_EXPLOSION,t,localize_explosion,valid);
		pblock->GetValue(PB_LOCALIZE_NOISE,t,localize_noise,valid);
		pblock->GetValue(PB_SEED,t,rseed,valid);
		pblock->GetValue(PB_FALLOFF,t,falloff,valid);

		density /= 50.0f;
		if (scale != 0.0f) scale = 1.0f/scale;
		if (stretch !=0.0f) stretch = 1.0f/stretch;
		
		if ((explode && (phase<0.0f || phase>300.0f)) && (!localize_explosion)){
			// Outside of explosion.
			rfact   = 0.0f;
			density = 0.0f;

		} else if ((explode)&& (!localize_explosion)) {
			// In explosion			

			// First compute radius factor			
			rfact = (float)sqrt(phase/100.0f);

			// next density
			if (phase>100.0f) {
				float u = 1.0f - (phase-100.0f)/200.0f;				
				density *= (u*u * (3.0f - 2.0f*u));
				}

			// Interpolate to smoke color
			if ((smoke) && (!localize_explosion)) {
				if (phase>200.0f) {
					// Solid smoke
					color1 = color2 = color3;
				} else if (phase>100.0f) {
					// Interpolate to smoke color
					float u = (phase-100.0f)/100.0f;
					u = (u*u * (3.0f - 2.0f*u));
					color1 += color3*u - color1*u;
					color2 += color3*u - color2*u;
					}
				}

			// Convert phase to a reasonable value and
			// add in fury factor
			phase = float(sqrt(phase/75.0f))/2.0f * 5.0f * fury;
		
		} 
		else if  ((explode)&& (localize_explosion) )
		{
//if localized explosion don't do anything since we will build a table of data later for each particle
		phase = float(sqrt(phase/75.0f))/2.0f * 5.0f * fury;
		}
		else {
			
			// Not doing an explosion
			rfact = 1.0f;
			phase = phase/300.0f * 5.0f;
			}
		}
	LeaveCriticalSection(&csect);
	}

int ExplodeAtmos::RenderBegin(TimeValue t,ULONG flags)
	{		




	return 0;
	}

int ExplodeAtmos::RenderEnd(TimeValue t)
	{	
	sources.Resize(0);
	return 0;
	}


ExplodeSource::ExplodeSource(
		INode *node,
		float drift, float rfact,
		TimeValue t, Interval &valid,
		int rseed)
	{
	int seed;
//	tm = Inverse(node->GetNodeTM(t,&valid));
	radius = 50.0f;
	hemi = 0;
	seed = 0;
	seed = rseed;
	radius   *= rfact;
	radius2   = radius*radius;
	hemiRange = radius/5.0f;
	hemiLen   = 2.0f*hemiRange;
	srand(seed);
	seedPt.x  = float(rand())/float(RAND_MAX) * 1000.0f;
	seedPt.y  = float(rand())/float(RAND_MAX) * 1000.0f;
	seedPt.z  = float(rand())/float(RAND_MAX) * 1000.0f;
	seedPt.z -= drift;


	ObjectState os = node->EvalWorldState(t);
		if (os.obj->SuperClassID()==GEOMOBJECT_CLASS_ID) {
			obj = (GeomObject*)os.obj;
		} else {
			obj = NULL;
			}

	// Back transform the ray into object space.		
	Matrix3 obtm  = node->GetObjectTM(t);
	tm = Inverse(obtm);



	}

void ExplodeAtmos::Update(TimeValue t, Interval& valid)
	{		
	UpdateCaches(t);
	sources.Resize(0);
	for (int i=0; i<nodes.Count(); i++) {
		if (!nodes[i]) continue;
//		ObjectState os = nodes[i]->EvalWorldState(t);

		ExplodeSource es(
				nodes[i],
				drift, rfact,
				t, valid,rseed);
			sources.Append(1,&es,10);
		}



	}

// Basically a turbulence function
inline float NoiseFunc(
		Point3 p,float phase,float levels,int invert) {
	float sum = 0.0f;
	float l,f = 1.0f;
	for (l = levels; l>=1.0f; l-=1.0f) {
		sum += (float)fabs(noise4(p*f,phase))/f;
		f *= 2.0f;
		}
	if (l>0.0f)
		sum += l*(float)fabs(noise4(p*f,phase))/f;
	if (invert) {
		sum = 0.5f-sum;
		return sum>0.0f?sum:0.0f;
	} else {
		return sum;
		}
	}


inline BOOL IntersectSphere(
		Ray &ray,float &t0,float &t1,float r)
	{	
	float a, b, c, ac4, b2;
	float root;	

	a = DotProd(ray.dir,ray.dir);
	b = DotProd(ray.dir,ray.p) * 2.0f;
	c = DotProd(ray.p,ray.p) - r*r;
	
	ac4 = 4.0f * a * c;
	b2 = b*b;

	if (ac4 > b2) 
		{
		return FALSE;
		}
	
	root = float(sqrt(b2-ac4));
	t0 = (-b + root) / (2.0f * a);
	t1 = (-b - root) / (2.0f * a);
	if (t0 > t1) {float temp=t0;t0=t1;t1=temp;}

	


	return TRUE;
	}

void ExplodeAtmos::TraceExplosion(
		ExplodeSource &src, Ray ray, Ray oray, float len,
		Color &c, float &o,
		TimeValue t)
	{
	float t0, t1, dist, dt, opac, u, n;
	Point3 dpt, pt, npt;
//	int ct = 0;
	Point3 norm;


		// Get the object from the node
/*	ObjectState os = nodes[ct]->EvalWorldState(t);
		if (os.obj->SuperClassID()==GEOMOBJECT_CLASS_ID) {
			obj = (GeomObject*)os.obj;
		} else {
			return;
			}

		// Back transform the ray into object space.		*/

//	Matrix3 obtm  = nodes[ct]->GetObjectTM(t);
//	Matrix3 iobtm = Inverse(obtm);
	Matrix3 iobtm = src.tm;
	BOOL done;
	Tab <float> f;
	Tab <float> b;

	f.ZeroCount();
	b.ZeroCount();

	ray.p   = iobtm * ray.p;
	ray.dir = VectorTransform(iobtm, ray.dir);

	oray = ray;
	done = FALSE;


//comput forward collisions;
	t1 = 0.0f;
	while (!done)
		{
		if (src.obj->IntersectRay(t,oray,t0,norm)) {
			t1 += t0;
		    f.Append(1,&t1,1);
			oray.p += oray.dir * (t0+0.001f);
			}
			else
			{
			done = TRUE;
			}
		}
//compute backward collisions;
	oray.p = ray.p + ray.dir*trace_back;
	oray.dir = -1.0f * ray.dir;
	done = FALSE;
	
	t1 = 0.0f;
	while (!done)
		{
		if (src.obj->IntersectRay(t,oray,t0,norm)) {
			if (t0<trace_back)
				{
				t1 += t0;
				b.Append(1,&t1,1);
				oray.p += oray.dir * (t0+0.001f);
				}
				else
				{
				done = TRUE;
				}
			}
			else
			{
			done = TRUE;
			}
		}

		// See if we hit the object

		if (b.Count() == 0)
			return;
		else if (b.Count() == (f.Count()+1))
			{
			t0 =0.0f;
			f.Insert(0,1,&t0);
			}

		if (b.Count() != f.Count())
			{	
			return;
			}
		else for (int k = 0; k<f.Count(); k++)
			{
			t0 = f[k];
			t1 = b[f.Count()-k-1];

			t1 = trace_back -t1;
//DebugPrint("Hit other wayt0 %f t1 %f\n",t0,t1);
	// We may be inside the sphere
//			if (t0<0.0f) t0 = 0.0f;
//			if (t1>len)  t1 = len;

	// The sphere may be in front, out of reach
//			if (t0>len) return;

	// The sphere may be behind us
//			if (t1<0.0f) return;

//DebugPrint("Interscted ray \n");
	 



			dist = t1-t0;
			dt   = dist/float(samples);
			pt   = ray.p + t0*ray.dir;
			dpt  = ray.dir * dt;
//DebugPrint("t0 %f t1 %f dist %f size\n",t0,t1,dist,size);

	// March along the ray
			for (int i=0; i<samples; i++,pt += dpt) {
		
				float p;
				if (object_size >=0.0f)
					p = dist/object_size;
					else p = 0.0f;
				
				if (p>falloff) 
					p = 1.0f;
					else p = p/falloff;

				u = 1.0f-p;					
				npt = pt + src.seedPt ; // offset by seed point
				npt.z *= stretch;      // apply stretch factor
				npt   *= scale;        // apply scaling
				n = NoiseFunc(npt,phase,levels,invert);
		
		// Make the noise more sparse as we reach the outer
		// parts of the explosion.
				if (u>regularity) {
					n -= u-regularity;
					if (n<0.0f) n = 0.0f;
					}

		// Compute opacity of this segment
				opac = (1.0f-u)*n*dt*density;

		// Add to color
				c += (color1*n + (1.0f-n)*color2)*opac;

		// Add to opacity
				o += opac;

				}

			}




	}

void ExplodeAtmos::Shade(
		ShadeContext& sc,const Point3& p0,const Point3& p1,
		Color& color, Color& trans, BOOL isBG)
	{
	if (sc.ProjType()==PROJ_PARALLEL) return;
	if (!sources.Count()) return;
	UpdateCaches(sc.CurTime());	

	if (density<=0.0f) return;

	Point3 wp0, wp1, v;
	float len;
	Ray ray,oray;

	// Setup the ray
	wp0   = sc.PointTo(p0,REF_WORLD);
	wp1   = sc.PointTo(p1,REF_WORLD);
	ray.p = wp0;
	v     = wp1-wp0;
	len   = Length(v);
	if (len==0.0f) return;
	ray.dir = v/len;


//	oray.p = wp1;
//	v     = wp0-wp1;
//	len   = Length(v);
//	if (len==0.0f) return;
//	oray.dir = v/len;



	// Start out color is clear and black
	Color c;
	c.Black();
	float o = 0.0f;

	int StartOff= 0;

	// Trace each explosion sphere
	for (int i=0; i<sources.Count(); i++) {




		TraceExplosion(sources[i],ray,oray,len,c,o,sc.CurTime());
		}

	// Exponentiate
	c.r = 1.0f - (float)exp(-c.r);
	c.g = 1.0f - (float)exp(-c.g);
	c.b = 1.0f - (float)exp(-c.b);
	o   = 1.0f - (float)exp(-o);
	
	// Combine with incoming color.
	color += c-(color*o);
	trans *= 1.0f-o;
	}

