/**********************************************************************
 *<
	FILE: explode.cpp

	DESCRIPTION: A fire/explosion particle atmospheric effect

	CREATED BY: Peter Watje	

	HISTORY: 12-25-96

 *>	Copyright (c) 1996 Peter Watje, All Rights Reserved.
 **********************************************************************/

#include "inferno.h"
#include "imtl.h"
#include "render.h"  
#include <bmmlib.h>
#include "iparamm.h"
#include "texutil.h"
//#include "gizmo.h"
//#include "expgizmo.h"
#include "istdplug.h"
#include "Simpobj.h"



static Class_ID volumeExplode(0x104d4ed7, 0xb43789f);

#define EXPLODE_CLASSNAME GetString(IDS_RB_COMBUSTION)

#define PBLOCK_REF	0
#define COLORMAP_REF	1
#define OBJECT_REF	2

#define A_RENDER			A_PLUGIN1

class NullView: public View {
	public:
		Point2 ViewToScreen(Point3 p) { return Point2(p.x,p.y); }
		NullView() { worldToView.IdentityMatrix(); screenW=640.0f; screenH = 480.0f; }
	};


/*
// The RenderLight is the container for each light in the scene.
class RenderLight {
public:
	// Constructor for normal lights
//	RenderLight(INode* node, MtlBaseLib* mtls);
	// Alternative constructor for default lights
//	RenderLight(DefaultLight* light);
	~RenderLight();


	RenderLight(INode* node);
//	void Update(TimeValue t, CJRenderer* renderer);
//	void UpdateViewDepParams(Matrix3 world2cam);

	INode* node;
	ObjLightDesc* desc;
	LightObject *light;
};

*/


class LightDescImp: public LightDesc {
	public:
	Point3 pos;
	Color col;
    BOOL Illuminate(ShadeContext& sc, Point3& normal, Color& color, Point3 &dir, float &dot_nl) {
		dir = Normalize(pos-sc.P());
		dot_nl = DotProd(normal,dir);
		color = col;
		return 1;		
		}
	};



class SCTex: public ShadeContext {
	public:
	float tiling;
	float scale;
	Color ambientLight;
	LightDescImp* lights[2];
	Point3 uvw,duvw,norm,view;
	IPoint2 scrPos;
	TimeValue curTime;
	Renderer *GetRenderer() { return NULL; }
	BOOL 	  InMtlEditor() { return TRUE; }
	LightDesc* Light(int n) { return lights[n]; }
	int ProjType() { return 1;} // returns: 0: perspective, 1: parallel
	int FaceNumber() { return 0; }
	TimeValue CurTime() { return curTime; }
	Point3 Normal() { return norm; }  	// interpolated normal
	void SetNormal(Point3 p) { norm = p;} 	// for perturbing normal
	Point3 GNormal() { return norm;} 	// geometric (face) normal
	Point3 ReflectVector() { return Point3(0,0,1); }
	Point3 RefractVector(float ior) {return Point3(0,0,1);	}
    Point3 CamPos() { return Point3(0,0,0); }			// camera position
	Point3 V() { return view; }       	// Unit view vector: from camera towards P 
	void SetView(Point3 v) { view =v; }
	Point3 P() { return scale*uvw; }			// point to be shaded in camera space;
	Point3 DP() { return scale*duvw; }   		// deriv of P, relative to pixel, for AA
	Point3 PObj() { return scale*uvw; }					  	// point in obj coords
	Point3 DPObj() { return scale*duvw; }   	// deriv of PObj, rel to pixel, for AA
	Box3 ObjectBox(); 	 			 	// Object extents box in obj coords
	Point3 PObjRelBox();   				// Point rel to obj box [-1 .. +1 ] 
	Point3 DPObjRelBox();  				// Point rel to obj box [-1 .. +1 ] 
	Point3 UVW(int chan) { return uvw;	};
   	Point3 DUVW(int chan) {	return duvw;	}
	Point3 UVW() { return uvw;	};
   	Point3 DUVW() {	return duvw;	}
	void DPdUVW(Point3 dP[3],int chan); 			// Bump vectors for UVW: in Camera space
	void DPdUVW(Point3 dP[3]); 			// Bump vectors for UVW: in Camera space
	AColor EvalEnvironMap(Texmap *map, Point3 viewd) {
		AColor rcol;
		return rcol;
		}

	void ScreenUV(Point2& uv, Point2 &duv); // screen coordinate
	IPoint2 SCTex::ScreenCoord() {return scrPos;}

	Point3 PointTo(const Point3& p, RefFrame ito) { return p; }
	Point3 PointFrom(const Point3& p, RefFrame ifrom) { return p; } 
	Point3 VectorTo(const Point3& p, RefFrame ito) { return p; } 
	Point3 VectorFrom(const Point3& p, RefFrame ifrom){ return p; } 
	void GetBGColor(Color &bgcol, Color& transp, BOOL fogBG=TRUE) {	}
	SCTex();
	void SetTiling(float t) { tiling = t; }
	};


SCTex::SCTex() {
	tiling = 1.0f;
	mtlNum = 0; 
	doMaps = TRUE;
	filterMaps = TRUE;
	shadow = FALSE;
	backFace = FALSE;
	curTime = 0;
	norm = Point3(0,0,1);
	view = Point3(0,0,-1);
	ResetOutput();
	}

Box3 SCTex::ObjectBox() {
	return Box3(Point3(0,0,0),Point3(scale,scale,scale));
	}

Point3 SCTex::PObjRelBox() {
	Point3 q;
	Point3 p = PObj();
	Box3 b = ObjectBox();
	q.x = 2.0f*(p.x-b.pmin.x)/(b.pmax.x-b.pmin.x) - 1.0f;
	q.y = 2.0f*(p.y-b.pmin.y)/(b.pmax.y-b.pmin.y) - 1.0f;
	q.z = 2.0f*(p.z-b.pmin.z)/(b.pmax.z-b.pmin.z) - 1.0f;
	return q;
	}

void SCTex::ScreenUV(Point2& uv, Point2 &duv) {
	uv.x = uvw.x;
	uv.y = uvw.x;
	duv.x = duvw.x;
	duv.y = duvw.y;
	}

Point3 SCTex::DPObjRelBox() {
	return Point3(0,0,0);
	}

// Bump vectors for UVW: in Camera space
void SCTex::DPdUVW(Point3 dP[3],int chan) { 
	dP[0] = dP[1] = dP[2] = Point3(0,0,0);
	}
void SCTex::DPdUVW(Point3 dP[3]) { 
	dP[0] = dP[1] = dP[2] = Point3(0,0,0);
	}





class ExplodeDlgProc;


class ParticleData {
	public:
		float p;
		Point3 lcolor1,lcolor2,lcolor3;
		float lrfact,ldensity;
		float lradius, lradius2;

	} ;

class ExplodeSource {
	public:
		Matrix3 tm;
		int hemi;
		Point3 seedPt;
		Object *objs;
		INode *node;
		float radius, radius2, hemiRange, hemiLen;
		ExplodeSource(INode *node,Object *obj,
			float drift, float rfact,
			TimeValue t, Interval &valid, int rseed);
		ExplodeSource();
	};

class ExplodeAtmos : public Atmospheric {
	public:
		// Parameters
		IParamBlock *pblock;  // Ref #0
		Tab<INode*> nodes;	  // Ref #1-n

		// Caches
		Tab<ExplodeSource> sources;		
		float stretch, regularity, drift, rfact;
		float density, scale, phase, levels, fury;
		float particle_size;
		int particle_life,density_path,size_path;
		int localize_explosion;
		int localize_noise;
		int rseed;
		Color color1, color2, color3;
		int samples, invert, explode, smoke;

		
		IRendParams *irp;
		Texmap *ColorMap;   // a reference
		void AssignColorMap(HWND hWnd);
		AColor ExplodeAtmos::GetPixel(float u, float v);
		void EnumAuxFiles(NameEnumCallback& nameEnum, DWORD flags) {
			if (ColorMap) ColorMap->EnumAuxFiles(nameEnum,flags);
			}


		TimeValue DataTime;

		Tab<ParticleData> PData;
		Interval valid;		
		CRITICAL_SECTION csect;

		static ExplodeDlgProc *dlg;

		ExplodeAtmos();
		~ExplodeAtmos() {DeleteCriticalSection(&csect);}

		void UpdateCaches(TimeValue t);

		// Animatable/Reference
		int NumSubs() {return 1;}
		Animatable* SubAnim(int i) {return pblock;}
		TSTR SubAnimName(int i) {return _T("");}
		int NumRefs();
		RefTargetHandle GetReference(int i);
		void SetReference(int i, RefTargetHandle rtarg);
		Class_ID ClassID() {return volumeExplode;}
		void GetClassName(TSTR& s) {s=EXPLODE_CLASSNAME;}
		void DeleteThis() {delete this;}
		RefResult NotifyRefChanged(Interval changeInt, RefTargetHandle hTarget, 
	         PartID& partID,  RefMessage message);
		IOResult Load(ILoad *iload);
		IOResult Save(ISave *isave);

		// Atmospheric
		TSTR GetName() {return EXPLODE_CLASSNAME;}
		AtmosParamDlg *CreateParamDialog(IRendParams *ip);
		int RenderBegin(TimeValue t, ULONG flags);
		int RenderEnd(TimeValue t);
		void Update(TimeValue t, Interval& valid);
		void Shade(ShadeContext& sc,const Point3& p0,const Point3& p1,Color& color, Color& trans, BOOL isBG);

		void TraceExplosion(
			ExplodeSource &src, Ray ray, float len,
			Color &c, float &o,
			float cx, float cy, float cz,
			Point3 lcolor1, Point3 lcolor2,
			float ldensity,
			float lradius, float lradius2,
			Point3 RNoise);
	};

class ExplodeParamDlg : public AtmosParamDlg {
	public:
		ExplodeAtmos *atmos;
		IRendParams *ip;
		IParamMap *pmap;

		ExplodeParamDlg(ExplodeAtmos *a,IRendParams *i);
		Class_ID ClassID() {return volumeExplode;}
		ReferenceTarget* GetThing() {return atmos;}
		void SetThing(ReferenceTarget *m);		
		void DeleteThis();
	};


class ExplodeClassDesc:public ClassDesc {
	public:
	int 			IsPublic() { return 1; }
	void *			Create(BOOL loading) { return new ExplodeAtmos; }
	const TCHAR *	ClassName() { return EXPLODE_CLASSNAME; }
	SClass_ID		SuperClassID() { return ATMOSPHERIC_CLASS_ID; }
	Class_ID 		ClassID() { return volumeExplode; }
	const TCHAR* 	Category() { return _T("");  }
	};

static ExplodeClassDesc explodeCD;
ClassDesc* GetExplodeDesc() {return &explodeCD;}

#define PB_COLOR1			0
#define PB_COLOR2			1
#define PB_COLOR3			2
#define PB_FLAMETYPE		3
#define PB_STRETCH			4
#define PB_REGULARITY		5
#define PB_SCALE			6
#define PB_LEVELS			7
#define PB_DENSITY			8
#define PB_SAMPLES			9
#define PB_PHASE			10
#define PB_DRIFT			11
#define PB_EXPLODE			12
#define PB_SMOKE			13
#define PB_FURY				14
#define PB_PARTICLE_SIZE	15
#define PB_PARTICLE_LIFE	16
#define PB_DENSITY_PATH		17
#define PB_SIZE_PATH		18
#define PB_LOCALIZE_EXPLOSION 19
#define PB_LOCALIZE_NOISE	20
#define PB_SEED				21

static int typeIDs[] = {IDC_EXPLODE_FIREBALL,IDC_EXPLODE_TENDRIL};

static ParamUIDesc descParam[] = {
	
	// Color 1
	ParamUIDesc(PB_COLOR1,TYPE_COLORSWATCH,IDC_EXPLODE_COLOR1),

	// Color 2
	ParamUIDesc(PB_COLOR2,TYPE_COLORSWATCH,IDC_EXPLODE_COLOR2),

	// Color 3
	ParamUIDesc(PB_COLOR3,TYPE_COLORSWATCH,IDC_EXPLODE_COLOR3),

	// Type
	ParamUIDesc(PB_FLAMETYPE,TYPE_RADIO,typeIDs,2),

	// Stretch
	ParamUIDesc(
		PB_STRETCH,
		EDITTYPE_FLOAT,
		IDC_EXPLODE_STRETCH,IDC_EXPLODE_STRETCHSPIN,
		0.0f,999999999.0f,
		0.1f),	
	
	// Regularity
	ParamUIDesc(
		PB_REGULARITY,
		EDITTYPE_FLOAT,
		IDC_EXPLODE_REG,IDC_EXPLODE_REGSPIN,
		0.0f,1.0f,
		0.01f),
		
	// Scale
	ParamUIDesc(
		PB_SCALE,
		EDITTYPE_FLOAT,
		IDC_EXPLODE_SCALE,IDC_EXPLODE_SCALESPIN,
		0.0f,999999999.0f,
		0.01f),
	
	// Levels
	ParamUIDesc(
		PB_LEVELS,
		EDITTYPE_FLOAT,
		IDC_EXPLODE_LEVELS,IDC_EXPLODE_LEVELSSPIN,
		1.0f,10.0f,
		0.01f),

	// Density
	ParamUIDesc(
		PB_DENSITY,
		EDITTYPE_FLOAT,
		IDC_EXPLODE_DENISITY,IDC_EXPLODE_DENISITYSPIN,
		0.0f,999999999.0f,
		0.1f),	

	// Samples
	ParamUIDesc(
		PB_SAMPLES,
		EDITTYPE_INT,
		IDC_EXPLODE_SAMPLES,IDC_EXPLODE_SAMPLESSPIN,
		1.0f,100.0f,
		0.5f),		

	// Phase
	ParamUIDesc(
		PB_PHASE,
		EDITTYPE_FLOAT,
		IDC_EXPLODE_PHASE,IDC_EXPLODE_PHASESPIN,
		0.0f,999999999.0f,
		0.01f),
	
	// Drift
	ParamUIDesc(
		PB_DRIFT,
		EDITTYPE_FLOAT,
		IDC_EXPLODE_DRIFT,IDC_EXPLODE_DRIFTSPIN,
		0.0f,999999999.0f,
		0.01f),

	// Explode
	ParamUIDesc(PB_EXPLODE,TYPE_SINGLECHEKBOX,IDC_EXPLODE_ON),

	// Smoke
	ParamUIDesc(PB_SMOKE,TYPE_SINGLECHEKBOX,IDC_EXPLODE_SMOKE),
	
	// Fury
	ParamUIDesc(
		PB_FURY,
		EDITTYPE_FLOAT,
		IDC_EXPLODE_FURY,IDC_EXPLODE_FURYSPIN,
		0.0f,999999999.0f,
		0.01f),	

	// Particle Size
	ParamUIDesc(
		PB_PARTICLE_SIZE,
		EDITTYPE_FLOAT,
		IDC_PARTICLE_SIZE,IDC_PARTICLE_SIZESPIN,
		0.0f,999999.0f,
		1.0f),

	// Particle Life
	ParamUIDesc(
		PB_PARTICLE_LIFE,
		EDITTYPE_INT,
		IDC_PARTICLE_LIFE,IDC_PARTICLE_LIFESPIN,
		0.0f,999999.0f,
		1.0f),

	// Denisty Path
	ParamUIDesc(PB_DENSITY_PATH,TYPE_SINGLECHEKBOX,IDC_DPATH_CHECK),

	// Life Path
	ParamUIDesc(PB_SIZE_PATH,TYPE_SINGLECHEKBOX,IDC_SPATH_CHECK),

	// Localize explosion
	ParamUIDesc(PB_LOCALIZE_EXPLOSION,TYPE_SINGLECHEKBOX,IDC_LOCALIZE_EXPLOSION_CHECK),

	// Localize noise
	ParamUIDesc(PB_LOCALIZE_NOISE,TYPE_SINGLECHEKBOX,IDC_LOCALIZE_NOISE_CHECK),


	// seed
	ParamUIDesc(
		PB_SEED,
		EDITTYPE_INT,
		IDC_SEED,IDC_SEED_SPIN,
		0.0f,999999.0f,
		1.0f),


	};

#define PARAMDESC_LENGH 22

static ParamBlockDescID descVer1[] = {
	{ TYPE_POINT3, NULL, TRUE, 0 }, // color 1
	{ TYPE_POINT3, NULL, TRUE, 1 }, // color 2
	{ TYPE_POINT3, NULL, TRUE, 2 }, // color 3
	{ TYPE_INT, NULL, FALSE, 3 },	// type
	{ TYPE_FLOAT, NULL, TRUE, 4 },	// stretch
	{ TYPE_FLOAT, NULL, TRUE, 5 },	// regularity
	{ TYPE_FLOAT, NULL, TRUE, 6 },	// scale
	{ TYPE_FLOAT, NULL, TRUE, 7 },	// levels
	{ TYPE_FLOAT, NULL, TRUE, 8 },	// density	
	{ TYPE_INT, NULL, FALSE, 9 },	// samples	
	{ TYPE_FLOAT, NULL, TRUE, 10 },	// phase
	{ TYPE_FLOAT, NULL, TRUE, 11 },	// drift
	{ TYPE_INT, NULL, FALSE, 12 },	// explode
	{ TYPE_INT, NULL, FALSE, 13 },// smoke
	{ TYPE_FLOAT, NULL, FALSE, 14 },// fury
	{ TYPE_FLOAT, NULL, TRUE, 15 },// particle size
	{ TYPE_INT, NULL, TRUE, 16 },// particle life
	{ TYPE_INT, NULL, FALSE, 17 },// density path
	{ TYPE_INT, NULL, FALSE, 18 },// size path
	{ TYPE_INT, NULL, FALSE, 19 },// localize explosion
	{ TYPE_INT, NULL, FALSE, 20 },// localize noise
	{ TYPE_INT, NULL, FALSE, 21 }// seed
	};

#define CURRENT_DESCRIPTOR descVer1

#define PBLOCK_LENGTH	22

#define CURRENT_VERSION	1


//--- ExplodeDlgProc ----------------------------------------------------------

class ExplodeDlgProc : 
			public ParamMapUserDlgProc,
			public RendPickProc {
	public:
		ExplodeAtmos *atmos;
		IRendParams *ip;
		ICustButton *iPick, *iRemove;		
		ISpinnerControl *iFury;
		HWND hWnd;
		HFONT hFont;

		ExplodeDlgProc(ExplodeAtmos *v,IRendParams *i);
		
		void Init(HWND hWnd);		
		BOOL DlgProc(TimeValue t,IParamMap *map,HWND hWnd,UINT msg,WPARAM wParam,LPARAM lParam);
		void DeleteThis();
		void UpdateNames();		
		void SetStates(HWND hWnd);
		void SetupExplosion(HWND hWnd);
		void DoAbout(HWND hWnd);

		// From RendPickProc
		BOOL Pick(INode *node);
		void Remove();
		BOOL Filter(INode *node);
		void EnterMode() {iPick->SetCheck(TRUE);}
		void ExitMode() {iPick->SetCheck(FALSE);}
	};

ExplodeDlgProc::ExplodeDlgProc(ExplodeAtmos *v,IRendParams *i) 
	{
	atmos   = v;
	ip      = i;
	iPick   = NULL;
	iFury   = NULL;
	iRemove = NULL;	
	atmos->dlg=this;
	atmos->irp = i;
	}

void ExplodeDlgProc::DeleteThis() 
	{	
	atmos->dlg = NULL;
	ip->EndPickMode();
	ReleaseICustButton(iPick);
	ReleaseICustButton(iRemove);	
	ReleaseISpinner(iFury);
	DeleteObject(hFont);
	delete this;
	}

BOOL ExplodeDlgProc::Pick(INode *node)
	{
	if (atmos->nodes.Count() && atmos->nodes[0]==NULL) {
		atmos->ReplaceReference(OBJECT_REF,node);
	} else {
		atmos->nodes.SetCount(atmos->nodes.Count()+1);
		atmos->nodes[atmos->nodes.Count()-1] = NULL;
		atmos->ReplaceReference(OBJECT_REF+atmos->nodes.Count()-1,node);
		}
	UpdateNames();
	return TRUE;
	}

void ExplodeDlgProc::Remove()
	{
	int sel = SendMessage(GetDlgItem(hWnd,IDC_EXPLODE_OBJECTNAME),CB_GETCURSEL,0,0);
	if (sel!=CB_ERR) {
		int c=0;
		for (int i=0; i<atmos->nodes.Count(); i++) {
			if (atmos->nodes[i]) {
				if (sel==c) {
					atmos->DeleteReference(i+OBJECT_REF);
					atmos->nodes.Delete(i,1);
					UpdateNames();
					break;
					}
				c++;
				}
			}
		}
	}

BOOL ExplodeDlgProc::Filter(INode *node)
	{
	for (int i=0; i<atmos->nodes.Count(); i++) {
		if (atmos->nodes[i]==node) return FALSE;
		}
	ObjectState os = node->EvalWorldState(ip->GetTime());
//filter only particle objects
//	if (os.obj->GetInterface(I_PARTICLEOBJ)==NULL)
//		return FALSE;
//		else return TRUE;
		return TRUE;
	}

void ExplodeDlgProc::UpdateNames()
	{
	int c=0;
	SendMessage(GetDlgItem(hWnd,IDC_EXPLODE_OBJECTNAME),CB_RESETCONTENT,0,0);
	for (int i=0; i<atmos->nodes.Count(); i++) {
		if (atmos->nodes[i]) {
			c++;
			SendMessage(GetDlgItem(hWnd,IDC_EXPLODE_OBJECTNAME),
				CB_ADDSTRING,0,(LPARAM)(const TCHAR*)atmos->nodes[i]->GetName());
			}
		}
	SendMessage(GetDlgItem(hWnd,IDC_EXPLODE_OBJECTNAME),CB_SETCURSEL,0,0);
	if (c) {
		iRemove->Enable();
	} else {
		iRemove->Disable();
		}
	}

void ExplodeDlgProc::SetStates(HWND hWnd)
	{
	int explode;
	atmos->pblock->GetValue(PB_EXPLODE,0,explode,FOREVER);
	if (explode) {		
		iFury->Enable();
		EnableWindow(GetDlgItem(hWnd,IDC_EXPLODE_SMOKE),TRUE);
		EnableWindow(GetDlgItem(hWnd,IDC_EXPLODE_SETUPPHASE),TRUE);		
		EnableWindow(GetDlgItem(hWnd,IDC_EXPLODE_FURYLABEL),TRUE);		
	} else {		
		iFury->Disable();
		EnableWindow(GetDlgItem(hWnd,IDC_EXPLODE_SMOKE),FALSE);
		EnableWindow(GetDlgItem(hWnd,IDC_EXPLODE_SETUPPHASE),FALSE);		
		EnableWindow(GetDlgItem(hWnd,IDC_EXPLODE_FURYLABEL),FALSE);
		}
	}


void ExplodeDlgProc::Init(HWND hWnd)
	{
	this->hWnd = hWnd;
	iPick = GetICustButton(GetDlgItem(hWnd,IDC_EXPLODE_PICK));
	iPick->SetType(CBT_CHECK);
	iPick->SetHighlightColor(GREEN_WASH);
	iRemove = GetICustButton(GetDlgItem(hWnd,IDC_EXPLODE_REMOVE));	
	iFury   = GetISpinner(GetDlgItem(hWnd,IDC_EXPLODE_FURYSPIN));
	hFont   =  CreateFont(24,0,0,0,FW_BOLD,0,0,0,0,0,0,0, VARIABLE_PITCH | FF_SWISS, _T(""));
	SendDlgItemMessage(hWnd,IDC_INFERNO_TITLE,WM_SETFONT,(WPARAM)hFont,TRUE);
	UpdateNames();
	}


BOOL ExplodeDlgProc::DlgProc(
		TimeValue t,IParamMap *map,HWND hWnd,
		UINT msg,WPARAM wParam,LPARAM lParam)
	{
	switch (msg) {
		case WM_INITDIALOG:
			Init(hWnd);
			SetStates(hWnd);

			if (atmos->ColorMap)
				SetDlgItemText( hWnd,IDC_COLOR_BUTTON,atmos->ColorMap->GetFullName());
			else 
				SetDlgItemText( hWnd,IDC_COLOR_BUTTON,GetString(IDS_DB_NONE));

			break;

		case WM_COMMAND:
			switch (LOWORD(wParam)) {
				case IDC_EXPLODE_PICK:
					ip->SetPickMode(this);
					break;

				case IDC_EXPLODE_REMOVE:
					Remove();
					break;

				case IDC_EXPLODE_SETUPPHASE:
					SetupExplosion(hWnd);
					break;

				case IDC_EXPLODE_ON:
					SetStates(hWnd);
					break;

				case IDC_EXPLODE_ABOUT:
					DoAbout(hWnd);
					break;
				case IDC_COLOR_BUTTON:
					atmos->AssignColorMap(hWnd);
					break;

				}
			break;
		}
	
	return FALSE;
	}

static BOOL CALLBACK SetupExplodeDlgProc(
		HWND hWnd, UINT msg, WPARAM wParam, LPARAM lParam)
	{
	static Interval *range = NULL;

	switch (msg) {
		case WM_INITDIALOG: {
			range = (Interval*)lParam;
			ISpinnerControl *spin;
			spin = GetISpinner(GetDlgItem(hWnd,IDC_EXPLODE_STARTSPIN));
			spin->LinkToEdit(GetDlgItem(hWnd,IDC_EXPLODE_START),EDITTYPE_TIME);
			spin->SetLimits(TIME_NegInfinity,TIME_PosInfinity, FALSE);
			spin->SetValue(range->Start(),FALSE);
			spin->SetScale(10.0f);
			ReleaseISpinner(spin);

			spin = GetISpinner(GetDlgItem(hWnd,IDC_EXPLODE_ENDSPIN));
			spin->LinkToEdit(GetDlgItem(hWnd,IDC_EXPLODE_END),EDITTYPE_TIME);
			spin->SetLimits(TIME_NegInfinity,TIME_PosInfinity, FALSE);
			spin->SetValue(range->End(),FALSE);
			spin->SetScale(10.0f);
			ReleaseISpinner(spin);

			CenterWindow(hWnd, GetParent(hWnd));
			break;
			}

		case WM_COMMAND:
			switch (LOWORD(wParam)) {
				case IDOK: {
					ISpinnerControl *spin;
					TimeValue start, end;
					spin  = GetISpinner(GetDlgItem(hWnd,IDC_EXPLODE_STARTSPIN));
					start = spin->GetIVal();
					ReleaseISpinner(spin);
					spin  = GetISpinner(GetDlgItem(hWnd,IDC_EXPLODE_ENDSPIN));
					end   = spin->GetIVal();
					ReleaseISpinner(spin);
					range->Set(start,end);
					EndDialog(hWnd,1);
					break;
					}

				case IDCANCEL:
					EndDialog(hWnd,0);
					break;
				}
			break;

		default:
			return FALSE;
		}
	return TRUE;
	}

void ExplodeDlgProc::SetupExplosion(HWND hWnd)
	{
	static Interval iv(0,16000);
	if (!DialogBoxParam(
		hInstance,
		MAKEINTRESOURCE(IDD_EXPLODE_SETUP),
		hWnd,
		SetupExplodeDlgProc,
		(LONG)&iv)) return;	
	
	// Create a fresh bezier float controller.
	Control *cont = (Control*)CreateInstance(
		CTRL_FLOAT_CLASS_ID,
		Class_ID(HYBRIDINTERP_FLOAT_CLASS_ID,0));
	if (!cont) return;
	IKeyControl *ikey = GetKeyControlInterface(cont);
	if (!ikey) {
		cont->DeleteThis();
		return;
		}
	
	// Setup a key structure with slow in and fast out tangents
	IBezFloatKey key;
	SetInTanType(key.flags,BEZKEY_SLOW);
	SetOutTanType(key.flags,BEZKEY_FAST);
	key.intan = key.outtan = 0.0f;

	// Make the first key
	key.val  = 0.0f;
	key.time = iv.Start();
	ikey->AppendKey(&key);

	// Make the second key
	key.val  = 300.0f;
	key.time = iv.End();
	ikey->AppendKey(&key);

	// Sort the table
	ikey->SortKeys();

	// Replace the controller in the param block with this controller.
	atmos->pblock->
		SetController(PB_PHASE,cont,FALSE);
	}

static BOOL CALLBACK AboutDlgProc(
		HWND hWnd, UINT msg, WPARAM wParam, LPARAM lParam)
	{	
	switch (msg) {
		case WM_INITDIALOG:			
			SendDlgItemMessage(hWnd,IDC_INFERNO_TITLE,WM_SETFONT,(WPARAM)lParam,TRUE);
			CenterWindow(hWnd, GetParent(hWnd));
			break;
			
		case WM_COMMAND:
			switch (LOWORD(wParam)) {
				case IDOK:
					EndDialog(hWnd,0);
					break;
				}
			break;

		default:
			return FALSE;
		}
	return TRUE;
	}

void ExplodeDlgProc::DoAbout(HWND hWnd)
	{
	DialogBoxParam(
		hInstance,
		MAKEINTRESOURCE(IDD_EXPLODE_ABOUT),
		hWnd,
		AboutDlgProc,
		(LPARAM)hFont);	
	}


//--- ExplodeParamDlg -------------------------------------------------------

ExplodeParamDlg::ExplodeParamDlg(ExplodeAtmos *a,IRendParams *i) 
	{
	atmos = a;
	ip    = i;	
	pmap  = CreateRParamMap(
		descParam,PARAMDESC_LENGH,
		atmos->pblock,
		i,
		hInstance,
		MAKEINTRESOURCE(IDD_EXPLODE_PARAMS),
		GetString(IDS_RB_COMBUSTPARAMS),
		0);	
	
	pmap->SetUserDlgProc(new ExplodeDlgProc(atmos,ip));
	}

void ExplodeParamDlg::SetThing(ReferenceTarget *m)
	{
	assert(m->ClassID()==atmos->ClassID());
	atmos = (ExplodeAtmos*)m;
	pmap->SetParamBlock(atmos->pblock);	
	if (atmos->dlg) {
		atmos->dlg->atmos = atmos;
		atmos->dlg->UpdateNames();
		atmos->dlg->SetStates(atmos->dlg->hWnd);
		}
	}

void ExplodeParamDlg::DeleteThis()
	{
	DestroyRParamMap(pmap);
	delete this;
	}



//--- ExplodeAtmos -------------------------------------------------------

ExplodeDlgProc *ExplodeAtmos::dlg = NULL;

#define CF(c) (float(c)/float(255))



void ExplodeAtmos::AssignColorMap(HWND hWnd) {	
	BOOL newMat, cancel;
	MtlBase *m = irp->DoMaterialBrowseDlg(
		hWnd,BROWSE_MAPSONLY|BROWSE_INCNONE,newMat,cancel);
	if (!cancel) {
		assert(!m || IsTex(m));
		ReplaceReference(COLORMAP_REF,m);
		if (ColorMap) {
			SetDlgItemText(hWnd,IDC_COLOR_BUTTON,ColorMap->GetFullName());
			if (newMat) 
				ColorMap->InitSlotType(MAPSLOT_TEXTURE);

		} else {
			SetDlgItemText(hWnd,IDC_COLOR_BUTTON,GetString(IDS_DB_NONE));
			}		
/*		if (m) {
			projector = TRUE;
			CheckDlgButton(hWnd,IDC_PROJECTOR, projector);
			}
*/
		NotifyDependents(FOREVER,0,REFMSG_SUBANIM_STRUCTURE_CHANGED);
		}
	}



AColor ExplodeAtmos::GetPixel(float u, float v)
	{

	AColor c;
	if (ColorMap) {
		SCTex shadeContext;
		shadeContext.scale = 1.0f;
		shadeContext.duvw  = Point3(0.001f,0.001f,0.0f);	
		shadeContext.scrPos.x = int(u);
		shadeContext.scrPos.y = int(v);
		shadeContext.uvw.x = u;
		shadeContext.uvw.y = v;
		shadeContext.uvw.z = 0.0f;
		c  = ColorMap->EvalColor(shadeContext);
		}
	return c;
	}



ExplodeAtmos::ExplodeAtmos()
	{
	InitializeCriticalSection(&csect);
	MakeRefByID(FOREVER, 0, 
		CreateParameterBlock(
			CURRENT_DESCRIPTOR, 
			PBLOCK_LENGTH, 
			CURRENT_VERSION));
	assert(pblock);		

	pblock->SetValue(PB_COLOR1,0,Point3(CF(252),CF(202),CF(  0)));
	pblock->SetValue(PB_COLOR2,0,Point3(CF(225),CF( 30),CF( 30)));	
	pblock->SetValue(PB_COLOR3,0,Point3(0.1,0.1,0.1));
	pblock->SetValue(PB_FLAMETYPE,0,0);
	pblock->SetValue(PB_STRETCH,0,1.0f);
	pblock->SetValue(PB_REGULARITY,0,0.2f);
	pblock->SetValue(PB_SCALE,0,35.0f);	
	pblock->SetValue(PB_LEVELS,0,3.0f);
	pblock->SetValue(PB_DENSITY,0,15.0f);	
	pblock->SetValue(PB_SAMPLES,0,15);			
	pblock->SetValue(PB_PHASE,0,0.0f);
	pblock->SetValue(PB_DRIFT,0,0.0f);
	pblock->SetValue(PB_EXPLODE,0,0);
	pblock->SetValue(PB_SMOKE,0,1);
	pblock->SetValue(PB_FURY,0,1.0f);
	pblock->SetValue(PB_PARTICLE_SIZE,0,20.0f);
	pblock->SetValue(PB_PARTICLE_LIFE,0,30);
	pblock->SetValue(PB_DENSITY_PATH,0,0);
	pblock->SetValue(PB_SIZE_PATH,0,0);
	pblock->SetValue(PB_LOCALIZE_EXPLOSION,0,0);
	pblock->SetValue(PB_LOCALIZE_NOISE,0,0);
	pblock->SetValue(PB_SEED,0,0);
	DataTime = -1;
	ColorMap = NULL;
	valid.SetEmpty();
	}

#define NUMOBJECTS_CHUNK	0x0010

IOResult ExplodeAtmos::Load(ILoad *iload)
	{
	Atmospheric::Load(iload);
	ULONG nb;
	int num;
	IOResult res = IO_OK;
	while (IO_OK==(res=iload->OpenChunk())) {
		switch (iload->CurChunkID()) {
			case NUMOBJECTS_CHUNK: {
				res = iload->Read(&num,sizeof(num),&nb);
				nodes.SetCount(num);
				for (int i=0; i<num; i++) nodes[i] = NULL;
				break;
				}
			}
		iload->CloseChunk();
		if (res!=IO_OK)  return res;
		}	
	
	return IO_OK;
	}

IOResult ExplodeAtmos::Save(ISave *isave)
	{

	Atmospheric::Save(isave);

	ULONG nb;
	int num = nodes.Count();
		
	isave->BeginChunk(NUMOBJECTS_CHUNK);
	isave->Write(&num,sizeof(num),&nb);
	isave->EndChunk();	
	
	return IO_OK;
	}

int ExplodeAtmos::NumRefs() 
	{
	return 2+nodes.Count();
	}

RefTargetHandle ExplodeAtmos::GetReference(int i) 
	{
	switch (i) {
		case PBLOCK_REF:	return pblock;		
		case COLORMAP_REF: return ColorMap;
		default:
			if (i>=OBJECT_REF) return nodes[i-OBJECT_REF];
			else return NULL;			
		}
	}

void ExplodeAtmos::SetReference(int i, RefTargetHandle rtarg) 
	{	
	switch (i) {
		case PBLOCK_REF:	pblock = (IParamBlock*)rtarg; break;
		case COLORMAP_REF: ColorMap = (Texmap *)rtarg;break;
		default:
			if (i>=OBJECT_REF) 
				nodes[i-OBJECT_REF] = (INode*)rtarg; 
			break;
		}
	}
			
RefResult ExplodeAtmos::NotifyRefChanged(
		Interval changeInt, RefTargetHandle hTarget,
		PartID& partID,  RefMessage message) 
	{
	switch (message) {
		case REFMSG_TARGET_DELETED: {
			for (int i=0; i<nodes.Count(); i++) {
				if (hTarget==nodes[i]) {
					nodes[i] = NULL;					
					}
				}
			if (dlg) dlg->UpdateNames();
			break;
			}

		case REFMSG_CHANGE:
			valid.SetEmpty();
			break;

		case REFMSG_NODE_NAMECHANGE:
			if (dlg) dlg->UpdateNames();
			break;

		case REFMSG_GET_PARAM_DIM: {
			GetParamDim *gpd = (GetParamDim*)partID;
			switch (gpd->index) {			
				case PB_COLOR1: 	gpd->dim = stdColor255Dim; break;
				case PB_COLOR2: 	gpd->dim = stdColor255Dim; break;
				case PB_COLOR3: 	gpd->dim = stdColor255Dim; break;
				default: 			gpd->dim = defaultDim;
				}
			return REF_STOP; 
			}

		case REFMSG_GET_PARAM_NAME: {
			GetParamName *gpn = (GetParamName*)partID;
			switch (gpn->index) {				
				case PB_COLOR1:		gpn->name = GetString(IDS_RB_INNERCOLOR); break;
				case PB_COLOR2:		gpn->name = GetString(IDS_RB_OUTERCOLOR); break;
				case PB_COLOR3:		gpn->name = GetString(IDS_RB_SMOKECOLOR); break;
				case PB_STRETCH:	gpn->name = GetString(IDS_RB_STRETCH); break;
				case PB_REGULARITY:	gpn->name = GetString(IDS_RB_REGULARITY); break;
				case PB_SCALE:		gpn->name = GetString(IDS_RB_FLAMESIZE); break;
				case PB_LEVELS:		gpn->name = GetString(IDS_RB_FLAMEDETAIL); break;
				case PB_DENSITY:	gpn->name = GetString(IDS_RB_DENSITY); break;				
				case PB_PHASE:		gpn->name = GetString(IDS_RB_PHASE); break;
				case PB_DRIFT:		gpn->name = GetString(IDS_RB_DRIFT); break;
				case PB_PARTICLE_SIZE:	gpn->name = GetString(IDS_RB_PARTICLE_SIZE); break;
				case PB_PARTICLE_LIFE:	gpn->name = GetString(IDS_RB_PARTICLE_LIFE); break;
				default:			gpn->name = _T("?????"); break;
				}
			return REF_STOP; 
			}
		}
	return REF_SUCCEED;
	}

AtmosParamDlg *ExplodeAtmos::CreateParamDialog(IRendParams *ip)
	{
	return new ExplodeParamDlg(this,ip);
	}

void ExplodeAtmos::UpdateCaches(TimeValue t)
	{		
//ebugPrint("Updating Caches\n");
	EnterCriticalSection(&csect);
	if (!valid.InInterval(t)) {
		valid = FOREVER;				
		pblock->GetValue(PB_COLOR1,t,color1,valid);
		pblock->GetValue(PB_COLOR2,t,color2,valid);
		pblock->GetValue(PB_COLOR3,t,color3,valid);
		pblock->GetValue(PB_FLAMETYPE,t,invert,valid);
		pblock->GetValue(PB_STRETCH,t,stretch,valid);
		pblock->GetValue(PB_REGULARITY,t,regularity,valid);
		pblock->GetValue(PB_SCALE,t,scale,valid);
		pblock->GetValue(PB_LEVELS,t,levels,valid);
		pblock->GetValue(PB_DENSITY,t,density,valid);
		pblock->GetValue(PB_SAMPLES,t,samples,valid);		
		pblock->GetValue(PB_PHASE,t,phase,valid);
		pblock->GetValue(PB_DRIFT,t,drift,valid);
		pblock->GetValue(PB_EXPLODE,t,explode,valid);
		pblock->GetValue(PB_SMOKE,t,smoke,valid);
		pblock->GetValue(PB_FURY,t,fury,valid);		
		pblock->GetValue(PB_PARTICLE_SIZE,t,particle_size,valid);		
		pblock->GetValue(PB_PARTICLE_LIFE,t,particle_life,valid);		
		pblock->GetValue(PB_DENSITY_PATH,t,density_path,valid);
		pblock->GetValue(PB_SIZE_PATH,t,size_path,valid);
		pblock->GetValue(PB_LOCALIZE_EXPLOSION,t,localize_explosion,valid);
		pblock->GetValue(PB_LOCALIZE_NOISE,t,localize_noise,valid);
		pblock->GetValue(PB_SEED,t,rseed,valid);

		density /= 50.0f;
		if (scale != 0.0f) scale = 1.0f/scale;
		if (stretch !=0.0f) stretch = 1.0f/stretch;
		
		if ((explode && (phase<0.0f || phase>300.0f)) && (!localize_explosion)){
			// Outside of explosion.
			rfact   = 0.0f;
			density = 0.0f;

		} else if ((explode)&& (!localize_explosion)) {
			// In explosion			

			// First compute radius factor			
			rfact = (float)sqrt(phase/100.0f);

			// next density
			if (phase>100.0f) {
				float u = 1.0f - (phase-100.0f)/200.0f;				
				density *= (u*u * (3.0f - 2.0f*u));
				}

			// Interpolate to smoke color
			if ((smoke) && (!localize_explosion)) {
				if (phase>200.0f) {
					// Solid smoke
					color1 = color2 = color3;
				} else if (phase>100.0f) {
					// Interpolate to smoke color
					float u = (phase-100.0f)/100.0f;
					u = (u*u * (3.0f - 2.0f*u));
					color1 += color3*u - color1*u;
					color2 += color3*u - color2*u;
					}
				}

			// Convert phase to a reasonable value and
			// add in fury factor
			phase = float(sqrt(phase/75.0f))/2.0f * 5.0f * fury;
		
		} 
		else if  ((explode)&& (localize_explosion) )
		{
//if localized explosion don't do anything since we will build a table of data later for each particle
		phase = float(sqrt(phase/75.0f))/2.0f * 5.0f * fury;
		}
		else {
			
			// Not doing an explosion
			rfact = 1.0f;
			phase = phase/300.0f * 5.0f;
			}
		}
	LeaveCriticalSection(&csect);
	}

int ExplodeAtmos::RenderBegin(TimeValue t,ULONG flags)
	{		




	return 0;
	}

int ExplodeAtmos::RenderEnd(TimeValue t)
	{	
	sources.Resize(0);
	return 0;
	}


ExplodeSource::ExplodeSource(
		INode *snode,Object *obj, 
		float drift, float rfact,
		TimeValue t, Interval &valid,
		int rseed)
	{
	int seed;
//	tm = Inverse(snode->GetNodeTM(t,&valid));
	tm = snode->GetNodeTM(t,&valid);
	radius = 50.0f;
	hemi = 0;
	seed = 0;
	seed = rseed;
	radius   *= rfact;
	radius2   = radius*radius;
	hemiRange = radius/5.0f;
	hemiLen   = 2.0f*hemiRange;
	srand(seed);
	seedPt.x  = float(rand())/float(RAND_MAX) * 1000.0f;
	seedPt.y  = float(rand())/float(RAND_MAX) * 1000.0f;
	seedPt.z  = float(rand())/float(RAND_MAX) * 1000.0f;
	seedPt.z -= drift;
	objs = obj;
	node = snode;
	}

void ExplodeAtmos::Update(TimeValue t, Interval& valid)
	{		
	UpdateCaches(t);
	sources.Resize(0);
	NullView nullView;


	for (int i=0; i<nodes.Count(); i++) {

		if (!nodes[i]) continue;
		ObjectState os = nodes[i]->EvalWorldState(t);
		ExplodeSource es(
				nodes[i],
				os.obj,
				drift, rfact,
				t, valid,rseed);
			sources.Append(1,&es,10);
		}

//if localized explosion build table of data for each particle so we won't have to recompute it for each ray later on
//	if  ( (explode)&& (localize_explosion) )
	if  (ColorMap != NULL)
		{
 		EnterCriticalSection(&csect);
		PData.ZeroCount();
		ParticleData TData;

		ColorMap->Update(t,FOREVER);

		for (int i=0; i<sources.Count(); i++) {

			BOOL needDel;
			NullView nullView;
			Mesh *mesh = ((GeomObject*)sources[i].objs)->GetRenderMesh(t,sources[i].node,nullView,needDel);

			int ct = mesh->numVerts;
	   		Point3 *VertList =  mesh->verts;

			TVFace	*TVFaceList = mesh->tvFace;
			Face *FaceList = mesh->faces;
			for (int j=0; j<ct; j++) 
				{
				Point3 lcolor1,lcolor2,lcolor3;
				float lrfact,ldensity;
				float lradius, lradius2;

				lrfact = rfact;
				ldensity = density;

				Point3 TVertA(0.0f,0.0f,0.0f);
				if ( mesh->numTVerts > 0)
					{

				 	for (int k=0;k<mesh->numFaces;k++)
						{
						if ((int) FaceList[k].v[0] == j)
							{
							TVertA = mesh->getTVert(TVFaceList[k].t[0]);
							k = mesh->numFaces+1;
							}
						else if ((int)FaceList[k].v[1] == j)
							{
							TVertA = mesh->getTVert(TVFaceList[k].t[1]);
							k = mesh->numFaces+1;
							}
						else if ((int)FaceList[k].v[2] == j)
							{
							TVertA = mesh->getTVert(TVFaceList[k].t[2]);
							k = mesh->numFaces+1;
							}
						}
					}
				AColor c;
				c = GetPixel(TVertA.x,TVertA.y);
//				lcolor1 = c;
//				lcolor2 = c;
//				lcolor3 = c;
				TData.lrfact = rfact;
				TData.ldensity = density;
				TData.lcolor1 = c;
				TData.lcolor2 = c;
				TData.lcolor3 = c;
				PData.Append(1,&TData,1);

				}

			if (needDel) delete mesh;
			}

		LeaveCriticalSection(&csect);
		}


	}

// Basically a turbulence function
inline float NoiseFunc(
		Point3 p,float phase,float levels,int invert) {
	float sum = 0.0f;
	float l,f = 1.0f;
	for (l = levels; l>=1.0f; l-=1.0f) {
		sum += (float)fabs(noise4(p*f,phase))/f;
		f *= 2.0f;
		}
	if (l>0.0f)
		sum += l*(float)fabs(noise4(p*f,phase))/f;
	if (invert) {
		sum = 0.5f-sum;
		return sum>0.0f?sum:0.0f;
	} else {
		return sum;
		}
	}


inline BOOL IntersectSphere(
		Ray &ray,float &t0,float &t1,float r, Point3 trans)
	{	
	float a, b, c, ac4, b2;
	float root;	

	ray.p +=trans;
	a = DotProd(ray.dir,ray.dir);
	b = DotProd(ray.dir,ray.p) * 2.0f;
	c = DotProd(ray.p,ray.p) - r*r;
	
	ac4 = 4.0f * a * c;
	b2 = b*b;

	if (ac4 > b2) 
		{
		ray.p -=trans;
		return FALSE;
		}
	
	root = float(sqrt(b2-ac4));
	t0 = (-b + root) / (2.0f * a);
	t1 = (-b - root) / (2.0f * a);
	if (t0 > t1) {float temp=t0;t0=t1;t1=temp;}

	
	ray.p -=trans;


	return TRUE;
	}

void ExplodeAtmos::TraceExplosion(
		ExplodeSource &src, Ray ray, float len,
		Color &c, float &o,
		float cx, float cy, float cz, 
		Point3 lcolor1, Point3 lcolor2,
		float ldensity,
		float lradius,
		float lradius2,
		Point3 RNoise)
	{
	float t0, t1, dist, dt, opac, u, n;
	Point3 dpt, pt, npt;
//	Ray TempRay;

	// If the radius is 0, then there's nothing to do
	if (lradius<=0.0f) return;

	// Transform the ray into object space;
	Matrix3 temp_matrix;
	Point3 trans(-cx,-cy,-cz);

//    TempRay = ray;
	temp_matrix = src.tm;
//	temp_matrix.PreTranslate(trans);
//	ray.p   = ray.p * temp_matrix;
//	ray.dir = VectorTransform(temp_matrix,ray.dir);

	// Intersect the ray with the explosion sphere

	if (!IntersectSphere(
		ray, t0, t1, lradius,trans)) return;

	// We may be inside the sphere
 	if (t0<0.0f) t0 = 0.0f;
	if (t1>len)  t1 = len;

	// The sphere may be in front, out of reach
	if (t0>len) return;

	// The sphere may be behind us
	if (t1<0.0f) return;

//DebugPrint("Interscted ray \n");
	 
	// Setup everything

//	TempRay.p   = TempRay.p * src.tm;
//	TempRay.dir = VectorTransform(src.tm,TempRay.dir);


	dist = t1-t0;
	dt   = dist/float(samples);
	pt   = ray.p + t0*ray.dir;
	dpt  = ray.dir * dt;

	// March along the ray
	for (int i=0; i<samples; i++,pt += dpt) {
		
		// Below the hemisphere range
//		if (src.hemi && pt.z<-src.hemiRange) continue;

		// Distance from the origin
		u = DotProd(pt+trans,pt+trans)/lradius2;		

		// Check hemisphere
//		if (src.hemi && pt.z<src.hemiRange) {			
			// Use the falloff due to the hemisphere if
			// it's larger.
//			float u2 = 1.0f-(pt.z + src.hemiRange)/src.hemiLen;
//			if (u2>u) u = u2;
//			}

		// Noise at this point
		if (localize_noise)
			npt = pt + trans + src.seedPt + RNoise; // offset by seed point
		else npt = pt + src.seedPt ; // offset by seed point
		npt.z *= stretch;      // apply stretch factor
		npt   *= scale;        // apply scaling
		n = NoiseFunc(npt,phase,levels,invert);
		
		// Make the noise more sparse as we reach the outer
		// parts of the explosion.
		if (u>regularity) {
			n -= u-regularity;
			if (n<0.0f) n = 0.0f;
			}

		// Compute opacity of this segment
		opac = (1.0f-u)*n*dt*ldensity;

		// Add to color
		c += (lcolor1*n + (1.0f-n)*lcolor2)*opac;

		// Add to opacity
		o += opac;
		}
	}

void ExplodeAtmos::Shade(
		ShadeContext& sc,const Point3& p0,const Point3& p1,
		Color& color, Color& trans, BOOL isBG)
	{
	if (sc.ProjType()==PROJ_PARALLEL) return;
	if (!sources.Count()) return;
	UpdateCaches(sc.CurTime());	


	int lightCount = sc.nLights;

	if (density<=0.0f) return;

	Point3 wp0, wp1, v;
	float len;
	Ray ray;

	// Setup the ray
	wp0   = sc.PointTo(p0,REF_WORLD);
	wp1   = sc.PointTo(p1,REF_WORLD);
	ray.p = wp0;
	v     = wp1-wp0;
	len   = Length(v);
	if (len==0.0f) return;
	ray.dir = v/len;

	// Start out color is clear and black
	Color c;
	c.Black();
	float o = 0.0f;

	int StartOff= 0;

	// Trace each explosion sphere
	for (int i=0; i<sources.Count(); i++) {

		BOOL needDel;
		NullView nullView;
		Mesh *mesh = ((GeomObject*)sources[i].objs)->GetRenderMesh(sc.CurTime(),sources[i].node,nullView,needDel);

		int ct = mesh->numVerts;
	   	Point3 *VertList =  mesh->verts;

		TVFace	*TVFaceList = mesh->tvFace;
		Face *FaceList = mesh->faces;



//		for (int j=0; j<sources[i].particles->parts.Count(); j++) {
		for (int j=0; j<ct; j++) {
//			if (sources[i].particles->parts.Alive(j))
				{
//				float p;
				Point3 lcolor1,lcolor2,lcolor3;
				float lrfact,ldensity;
				float lradius, lradius2;

				lrfact = rfact;
				ldensity = density;

//get uv
//get color
/*
				if (ColorMap == NULL)
					{
					lcolor1 = color1;
					lcolor2 = color2;
					lcolor3 = color3;
					}
				else
					{
					Point3 TVertA(0.0f,0.0f,0.0f);
				 	for (int k=0;k<mesh->numFaces;k++)
						{
						if ((int) FaceList[k].v[0] == j)
							{
							TVertA = mesh->getTVert(TVFaceList[k].t[0]);
							k = mesh->numFaces+1;
							}
						else if ((int)FaceList[k].v[1] == j)
							{
							TVertA = mesh->getTVert(TVFaceList[k].t[0]);
							k = mesh->numFaces+1;
							}
						else if ((int)FaceList[k].v[2] == j)
							{
							TVertA = mesh->getTVert(TVFaceList[k].t[0]);
							k = mesh->numFaces+1;
							}
						}
					AColor c;
					c = GetPixel(TVertA.x,TVertA.y);
					lcolor1 = c;
					lcolor2 = c;
					lcolor3 = c;
					}
*/

//				if ((explode) && (localize_explosion)){
				if (ColorMap != NULL){
// In localized explosion use table data for densty, etc
					lcolor1 = PData[StartOff].lcolor1;
					lcolor2 = PData[StartOff].lcolor2;
					lcolor3 = PData[StartOff++].lcolor3;
					}
				else
					{
					lcolor1 = color1;
					lcolor2 = color2;
					lcolor3 = color3;
					}


				lradius = particle_size;
				lradius   *= lrfact;
				lradius2   = lradius*lradius;

				Point3 RNoise;

				RNoise.x = 0.0f;
				RNoise.y = 0.0f;
				RNoise.z = 0.0f;
				if (localize_noise)
					{
					RNoise.x = 50.0f * j+5;
					RNoise.y = 50.0f * j+5;
					RNoise.z = 50.0f * j+5;

					}

				Point3 target;

				target = VertList[j] * sources[i].tm;

				TraceExplosion(sources[i],ray,len,c,o,
/*				sources[i].particles->parts.points[j].x,
				sources[i].particles->parts.points[j].y,
				sources[i].particles->parts.points[j].z,*/
				target.x,target.y,target.z,
				lcolor1,lcolor2,
				ldensity,
				lradius,lradius2,
				RNoise);
				}
			}
//		StartOff += sources[i].particles->parts.Count();
		}

	// Exponentiate
	c.r = 1.0f - (float)exp(-c.r);
	c.g = 1.0f - (float)exp(-c.g);
	c.b = 1.0f - (float)exp(-c.b);
	o   = 1.0f - (float)exp(-o);
	
	// Combine with incoming color.
	color += c-(color*o);
	trans *= 1.0f-o;
	}

