/**********************************************************************
 *<
	FILE: cross.cpp

	DESCRIPTION:  Skins cross section

	CREATED BY: Peter Watje

	HISTORY: created Spet 13, 1996

 *>	Copyright (c) 1996, All Rights Reserved.
 **********************************************************************/

#include "mods.h"
#include "iparamm.h"
#include "shape.h"
#include "spline3d.h"
#include "splshape.h"
#include "splshape.h"
#include "linshape.h"

//--- ExtrudeMod -----------------------------------------------------------

#define EDITSPL_CHANNELS (PART_GEOM|SELECT_CHANNEL|PART_SUBSEL_TYPE|PART_DISPLAY|PART_TOPO)

#define MIN_AMOUNT		float(-1.0E30)
#define MAX_AMOUNT		float(1.0E30)

#define SPLINETYPE 0
#define CIRCULAR 0

#define PATCH_OUTPUT 0

#define DEF_OUTPUT MESH_OUTPUT //PATCH_OUTPUT



#define PW_PATCH_TO_SPLINE1 0x55c93a3e
#define PW_PATCH_TO_SPLINE2 0x8466b2b

//#define DEBUG 1



class ExtrudeMod: public Modifier {
	
	protected:

		static IObjParam *ip;
		
	public:
		IParamBlock *pblock;
		static IParamMap *pmapParam;
		
		static int steps;
		static BOOL optimize,adaptive;
		static BOOL renderable;



		// From Animatable
		void DeleteThis() { delete this; }
		void GetClassName(TSTR& s) { s= GetString(IDS_RB_CROSSSECT); }  
		virtual Class_ID ClassID() { return Class_ID(PW_PATCH_TO_SPLINE1,PW_PATCH_TO_SPLINE2);}		
		RefTargetHandle Clone(RemapDir& remap = NoRemap());
		TCHAR *GetObjectName() { return GetString(IDS_RB_CROSSSECT); }
		IOResult Load(ILoad *iload);

		ExtrudeMod();
		virtual ~ExtrudeMod();

		ChannelMask ChannelsUsed()  { return EDITSPL_CHANNELS; }
		ChannelMask ChannelsChanged() { return EDITSPL_CHANNELS; }


//		ChannelMask ChannelsUsed()  { return PART_GEOM|PART_TOPO; }
		// Possible GOTCHA -- Modifiers like this one, which completely change the type of
		// object (shape -> Mesh or Patch) change ALL channels!  Be sure to tell the system!
//		ChannelMask ChannelsChanged() { return PART_ALL; }
		void ModifyObject(TimeValue t, ModContext &mc, ObjectState *os, INode *node);

//		void NotifyInputChanged(Interval changeInt, PartID partID, RefMessage message, ModContext *mc);
		Class_ID InputType() { return Class_ID(SPLINESHAPE_CLASS_ID,0); }
		
//		Class_ID InputType() {return genericShapeClassID;}
		Interval LocalValidity(TimeValue t);

		// From BaseObject
		BOOL ChangeTopology() {return TRUE;}
		IParamArray *GetParamBlock() {return pblock;}
		int GetParamBlockIndex(int id) {return id;}

		int NumRefs() {return 1;}
		RefTargetHandle GetReference(int i) {return pblock;}
		void SetReference(int i, RefTargetHandle rtarg) {pblock=(IParamBlock*)rtarg;}

 		int NumSubs() { return 1; }  
		Animatable* SubAnim(int i) { return pblock; }
		TSTR SubAnimName(int i) { return TSTR(GetString(IDS_RB_PARAMETERS));}		

		RefResult NotifyRefChanged( Interval changeInt,RefTargetHandle hTarget, 
		   PartID& partID, RefMessage message);

		CreateMouseCallBack* GetCreateMouseCallBack() {return NULL;} 

		void BeginEditParams(IObjParam *ip, ULONG flags,Animatable *prev);
		void EndEditParams(IObjParam *ip, ULONG flags,Animatable *next);

		void BuildSkin(TimeValue t,ModContext &mc, ObjectState * os);

		void UpdateUI(TimeValue t) {}
		Interval GetValidity(TimeValue t);
		ParamDimension *GetParameterDim(int pbIndex);
		TSTR GetParameterName(int pbIndex);
	};

class ExtrudeClassDesc:public ClassDesc {
	public:
	int 			IsPublic() { return 1; }
	void *			Create(BOOL loading = FALSE) { return new ExtrudeMod; }
	const TCHAR *	ClassName() { return GetString(IDS_RB_CROSSSECT); }
	SClass_ID		SuperClassID() { return OSM_CLASS_ID; }
	Class_ID		ClassID() { return  Class_ID(PW_PATCH_TO_SPLINE1,PW_PATCH_TO_SPLINE2); }
	const TCHAR* 	Category() { return GetString(IDS_RB_PATCH);}
	};

static ExtrudeClassDesc extrudeDesc;
extern ClassDesc* GetExtrudeModDesc() { return &extrudeDesc; }

IObjParam*		ExtrudeMod::ip          = NULL;
IParamMap *		ExtrudeMod::pmapParam = NULL;
int				ExtrudeMod::steps = 5;
BOOL			ExtrudeMod::optimize = 0;
BOOL			ExtrudeMod::adaptive = 0;
BOOL			ExtrudeMod::renderable = 0;
 

//--- Parameter map/block descriptors -------------------------------



#define PB_STEPS		0
#define PB_OPTIMIZE		1
#define PB_ADAPTIVE		2
#define PB_LENGTH		3
#define PB_RENDERABLE	4


//
//
// Parameters


static ParamUIDesc descParam[] = {
	// steps
   // Steps

	ParamUIDesc(
		PB_STEPS,
		EDITTYPE_POS_INT,
		IDC_SHAPESTEPS,IDC_SSTEPSSPINNER,
		0.0f,30000.0f,
		1.0f),

	// optimize
   ParamUIDesc(PB_OPTIMIZE,TYPE_SINGLECHEKBOX,IDC_OPTIMIZE),

	// adaptive
   ParamUIDesc(PB_ADAPTIVE,TYPE_SINGLECHEKBOX,IDC_ADAPTIVE),

   // radius

	ParamUIDesc(
		PB_LENGTH,
		EDITTYPE_POS_FLOAT,
		IDC_SHAPESTEPS2,IDC_SSTEPSSPINNER2,
		0.0f,30000.0f,
		1.0f),


   // renderable
   ParamUIDesc(PB_RENDERABLE,TYPE_SINGLECHEKBOX,IDC_RENDERABLE)


	};

#define PARAMDESC_LENGTH 5


static ParamBlockDescID descVer0[] = {
	{ TYPE_FLOAT, NULL, TRUE, 0 },
	{ TYPE_INT, NULL, TRUE, 1 },
	{ TYPE_INT, NULL, FALSE, 2 },
	{ TYPE_INT, NULL, FALSE, 3 } };

static ParamBlockDescID descVer1[] = {
	{ TYPE_FLOAT, NULL, TRUE, 0 },
	{ TYPE_INT, NULL, TRUE, 1 },
	{ TYPE_INT, NULL, FALSE, 2 },
	{ TYPE_INT, NULL, FALSE, 3 },
	{ TYPE_INT, NULL, FALSE, 4 } };

static ParamBlockDescID descVer2[] = {
	{ TYPE_FLOAT, NULL, TRUE, 0 },
	{ TYPE_INT, NULL, TRUE, 1 },
	{ TYPE_INT, NULL, FALSE, 2 },
	{ TYPE_INT, NULL, FALSE, 3 },
	{ TYPE_INT, NULL, FALSE, 4 },
	{ TYPE_INT, NULL, FALSE, 5 } };

static ParamBlockDescID descVer3[] = {
	{ TYPE_INT, NULL, FALSE, 0 },
	{ TYPE_INT, NULL, FALSE, 1 },
	{ TYPE_INT, NULL, FALSE, 2 },
	{ TYPE_FLOAT, NULL, TRUE, 3 }
	 };

static ParamBlockDescID descVer4[] = {
	{ TYPE_INT, NULL, FALSE, 0 },
	{ TYPE_INT, NULL, FALSE, 1 },
	{ TYPE_INT, NULL, FALSE, 2 },
	{ TYPE_FLOAT, NULL, TRUE, 3 },
	{ TYPE_INT, NULL, TRUE, 4 }
	 };

#define PBLOCK_LENGTH	5

// Array of old versions
static ParamVersionDesc versions[] = {
	ParamVersionDesc(descVer0,4,0),
	ParamVersionDesc(descVer1,5,1),	
	ParamVersionDesc(descVer2,6,2),
	ParamVersionDesc(descVer3,3,4)	
	};
#define NUM_OLDVERSIONS	4

// Current version
#define CURRENT_VERSION	5
static ParamVersionDesc curVersion(descVer4,PBLOCK_LENGTH,CURRENT_VERSION);

ExtrudeMod::ExtrudeMod()
	{
	MakeRefByID(FOREVER, 0, 
		CreateParameterBlock(descVer3, PBLOCK_LENGTH, CURRENT_VERSION));
	pblock->SetValue(PB_STEPS, TimeValue(0), steps);
	pblock->SetValue(PB_ADAPTIVE, TimeValue(0), adaptive);
	pblock->SetValue(PB_RENDERABLE, TimeValue(0), renderable);
	pblock->SetValue(PB_OPTIMIZE, TimeValue(0), optimize);
	pblock->SetValue(PB_LENGTH, TimeValue(0), 1.0f);
	
	}

ExtrudeMod::~ExtrudeMod()
	{	
	}

Interval ExtrudeMod::LocalValidity(TimeValue t)
	{
	// if being edited, return NEVER forces a cache to be built 
	// after previous modifier.
	if (TestAFlag(A_MOD_BEING_EDITED))
		return NEVER;  
//	Interval valid = GetValidity(t);	
	Interval valid = FOREVER;
	float i;
	pblock->GetValue(PB_LENGTH,t,i,valid);	
	return valid;

	return valid;
	}

RefTargetHandle ExtrudeMod::Clone(RemapDir& remap)
	{
	ExtrudeMod* newmod = new ExtrudeMod();	
	newmod->ReplaceReference(0,pblock->Clone(remap));
	BaseClone(this, newmod, remap);
	return(newmod);
	}

void ExtrudeMod::ModifyObject(TimeValue t, ModContext &mc, ObjectState * os, INode *node) 
	{	
//DebugPrint("Extrude modifying object\n");

	// Get our personal validity interval...
	Interval valid = GetValidity(t);
	// and intersect it with the channels we use as input (see ChannelsUsed)
	valid &= os->obj->ChannelValidity(t,TOPO_CHAN_NUM);
	valid &= os->obj->ChannelValidity(t,GEOM_CHAN_NUM);
	Matrix3 modmat,minv;
	
//	int output;


//Dump from spline to patch
		// Here is where all the fun stuff happens -- BuildPatchFromShape fills in the PatchObject's patch mesh,
		// then we stuff the PatchObject into the pipeline.
	
	

	BuildSkin(t, mc, os);

	


	os->obj->SetChannelValidity(TOPO_CHAN_NUM, valid);
	os->obj->SetChannelValidity(GEOM_CHAN_NUM, valid);
	os->obj->SetChannelValidity(TEXMAP_CHAN_NUM, valid);
	os->obj->SetChannelValidity(MTL_CHAN_NUM, valid);
	os->obj->SetChannelValidity(SELECT_CHAN_NUM, valid);
	os->obj->SetChannelValidity(SUBSEL_TYPE_CHAN_NUM, valid);
	os->obj->SetChannelValidity(DISP_ATTRIB_CHAN_NUM, valid);

//	os->obj = pat;

//	os->obj->UnlockObject();
//	os->obj->InvalidateGeomCache();

	}


class SimpleSplineDlgProc : public ParamMapUserDlgProc {
	private:
		ExtrudeMod *spl;
	public:
		SimpleSplineDlgProc(ExtrudeMod *s) { spl = s; }
		BOOL DlgProc(TimeValue t,IParamMap *map,HWND hWnd,UINT msg,WPARAM wParam,LPARAM lParam);
		void DeleteThis() { delete this; }
		void MaybeDisableControls(HWND hWnd);
	};
						 


void SimpleSplineDlgProc::MaybeDisableControls(HWND hWnd) {
	BOOL adaptive;
	spl->pblock->GetValue(PB_ADAPTIVE,0,adaptive,FOREVER);
	EnableWindow(GetDlgItem(hWnd,IDC_OPTIMIZE), !adaptive);
	EnableWindow(GetDlgItem(hWnd,IDC_SHAPESTEPS), !adaptive);
	EnableWindow(GetDlgItem(hWnd,IDC_SSTEPSSPINNER), !adaptive);
	}

BOOL SimpleSplineDlgProc::DlgProc(
		TimeValue t,IParamMap *map,
		HWND hWnd,UINT msg,WPARAM wParam,LPARAM lParam)
	{
	switch (msg) {
		case WM_INITDIALOG:
			MaybeDisableControls(hWnd);
			break;
		case WM_COMMAND:
			switch (LOWORD(wParam)) {
				case IDC_ADAPTIVE:
					MaybeDisableControls(hWnd);
					break;
				}
			break;
		}
	return FALSE;
	}


void ExtrudeMod::BeginEditParams( IObjParam *ip, ULONG flags,Animatable *prev )
	{
	this->ip = ip;

	TimeValue t = ip->GetTime();

		// Disable show end result.
	ip->EnableShowEndResult(FALSE);



	NotifyDependents(Interval(t,t), PART_ALL, REFMSG_BEGIN_EDIT);
	NotifyDependents(Interval(t,t), PART_ALL, REFMSG_MOD_DISPLAY_ON);

	SetAFlag(A_MOD_BEING_EDITED);

	pmapParam = CreateCPParamMap(
		descParam,PARAMDESC_LENGTH,
		pblock,
		ip,
		hInstance,
		MAKEINTRESOURCE(IDD_WELD),
		GetString(IDS_RB_PARAMETERS),
		0);	

	pmapParam->SetUserDlgProc(new SimpleSplineDlgProc(this));

	}

void ExtrudeMod::EndEditParams( IObjParam *ip, ULONG flags,Animatable *next)
	{
	this->ip = NULL;
	
	TimeValue t = ip->GetTime();

	NotifyDependents(Interval(t,t), PART_ALL, REFMSG_END_EDIT);
	NotifyDependents(Interval(t,t), PART_ALL, REFMSG_MOD_DISPLAY_OFF);
	ClearAFlag(A_MOD_BEING_EDITED);

	DestroyCPParamMap(pmapParam);

	// Save these values in class variables so the next object created will inherit them.
	pblock->GetValue(PB_STEPS, t, steps, FOREVER);
	pblock->GetValue(PB_OPTIMIZE, t, optimize, FOREVER);
	pblock->GetValue(PB_ADAPTIVE, t, adaptive, FOREVER);
	pblock->GetValue(PB_RENDERABLE, t, renderable, FOREVER);

	}

Interval ExtrudeMod::GetValidity(TimeValue t)
	{
//	float f;
//	int i;
	Interval valid = FOREVER;
	return valid;
	}

RefResult ExtrudeMod::NotifyRefChanged(
		Interval changeInt, 
		RefTargetHandle hTarget, 
   		PartID& partID, 
   		RefMessage message ) 
   	{
	switch (message) {
		case REFMSG_GET_PARAM_DIM: {
			GetParamDim *gpd = (GetParamDim*)partID;
			switch (gpd->index) {				
				case 0:
				default: gpd->dim = defaultDim; break;
				}			
			return REF_STOP; 
			}

		case REFMSG_GET_PARAM_NAME: {
			GetParamName *gpn = (GetParamName*)partID;
			switch (gpn->index) {				
				case PB_LENGTH:	gpn->name = GetString(IDS_LENGTH); break;
				default:		gpn->name = TSTR(_T("")); break;
				}
			return REF_STOP; 
			}
		}
	return REF_SUCCEED;
	}

ParamDimension *ExtrudeMod::GetParameterDim(int pbIndex)
	{
	return defaultDim;
	}

TSTR ExtrudeMod::GetParameterName(int pbIndex)
	{
	return TSTR(_T(""));
	}

IOResult ExtrudeMod::Load(ILoad *iload)
	{
	Modifier::Load(iload);
	iload->RegisterPostLoadCallback(
		new ParamBlockPLCB(versions,NUM_OLDVERSIONS,&curVersion,this,0));
	return IO_OK;
	}


// Quad patch layout
//
//   A---> ad ----- da <---D
//   |                     |
//   |                     |
//   v                     v
//   ab    i1       i4     dc
//
//   |                     |
//   |                     |
// 
//   ba    i2       i3     cd
//   ^					   ^
//   |                     |
//   |                     |
//   B---> bc ----- cb <---C
//
// vertices ( a b c d ) are in counter clockwise order when viewed from 
// outside the surface


void ExtrudeMod::BuildSkin(TimeValue t,ModContext &mc, ObjectState * os) {


SplineShape *shape = (SplineShape *)os->obj;

pblock->GetValue(PB_STEPS, t, steps, FOREVER);
pblock->GetValue(PB_OPTIMIZE, t, optimize, FOREVER);
pblock->GetValue(PB_ADAPTIVE, t, adaptive, FOREVER);
pblock->GetValue(PB_RENDERABLE, t, renderable, FOREVER);
float radius;
pblock->GetValue(PB_LENGTH,t,radius,FOREVER);	

theHold.Suspend();
shape->shape.steps = adaptive ? -1 : steps;
shape->shape.optimize = optimize;
shape->SetRenderable(renderable);
shape->SetThickness(t,radius);
theHold.Resume();

shape->shape.UpdateSels();	// Make sure it readies the selection set info
shape->shape.InvalidateGeomCache();

			


}
