/*===========================================================================*\
 | File: lazypoint.cpp
 |
 | Peter Watje June 16 1997
\*===========================================================================*/


#include "MAX.H"

// This is the include file generated by the resource compiler.
#include "spherify.h"

// The DLL instance handle
HINSTANCE hInstance;

float sign(float x) { return (x < 0.0f ? -1.0f : 1.0f); }

/*===========================================================================*\
 | Misc Defines
\*===========================================================================*/
#define PERCENT_PB_INDEX 0
#define PBLOCK_REF_INDEX 0
#define SPH_CLASS_ID1 0xDE17A34f
#define SPH_CLASS_ID2 0x8A41E2A9
#define PARAMETERS_NAME _T("Parameters")
#define MODIFIER_NAME _T("Lazy")
#define CLASSNAME _T("Lazy Mod")
#define SPHERIFY_CNAME _T("Lazy")
#define PERCENT_PARAM_NAME _T("Percent")
#define UNDO_NAME _T("Parameter Change")
#define CATEGORY_NONE _T("")

#define PBLOCK    0
#define SELF_NODE 1

/*===========================================================================*\
 | Class Definitions:
\*===========================================================================*/
class SpherifyMod: public Modifier {
	friend BOOL CALLBACK SpherifyDialogProc( HWND hDlg, UINT message, 
		WPARAM wParam, LPARAM lParam );

	private:
		IParamBlock *pblock;

		// Class vars
		static HWND hSpherifyParams;
		static ISpinnerControl *percentSpin;	

		// This is an inherited virtual method from ReferenceMaker.
		RefResult NotifyRefChanged( Interval changeInt,RefTargetHandle hTarget,
		   PartID& partID, RefMessage message );

	public:
		SpherifyMod();
		static IObjParam *iObjParams;
		INode *SelfNode;
		Tab<TimeValue> BackTimes;
		Tab<Matrix3> BackMatrices;
		Matrix3 CurrentMatrix;
										
		// This method access the parameter block.
		void SetPercent(TimeValue t, float f );
		float GetPercent(TimeValue t, Interval& valid = Interval(0,0));
		void UpdateUI(TimeValue t);

		ChannelMask ChannelsUsed()  {return PART_GEOM|PART_TOPO;}
		ChannelMask ChannelsChanged() {return PART_GEOM;}
		Class_ID InputType() { return defObjectClassID; }
 		BOOL ChangeTopology() {return FALSE;}

		void ModifyObject(TimeValue t, ModContext &mc, ObjectState * os, INode *node );

		Interval SphValidity(TimeValue t);

// ---- Inherited virtual methods from Animatable ----
		// This method deletes the instance from memory.
		void DeleteThis() { delete this; }
		// The name of the class for debugging
		void GetClassName(TSTR& s) { s= CLASSNAME; }  
		// This method returns the unique class ID.
		virtual Class_ID ClassID() { return Class_ID(SPH_CLASS_ID1, SPH_CLASS_ID2);}
		// Our super class id is the object space modifier super class.
		SClass_ID SuperClassID() { return OSM_CLASS_ID; }
		
		// These two methods start and stop editing the modifiers parameters.
		void BeginEditParams(IObjParam *ip, ULONG flags, Animatable *prev);
		void EndEditParams(IObjParam *ip,ULONG flags, Animatable *next);

		int NumSubs() { return 1; }  
		Animatable* SubAnim(int i) { return pblock; }
		TSTR SubAnimName(int i) { return PARAMETERS_NAME;}		

// ---- Inherited virtual methods from ReferenceMaker ----
		int NumRefs() { return 2;}
		RefTargetHandle GetReference(int i) {
			if (i==PBLOCK)	return pblock;
			else if (i==SELF_NODE)	return SelfNode;
			else return NULL;
			}
		void SetReference(int i, RefTargetHandle rtarg) {
			if (i==PBLOCK) pblock = (IParamBlock*)rtarg;
			else if (i==SELF_NODE) SelfNode = (INode*)rtarg;

			}

// ---- Inherited virtual methods from ReferenceTarget ----
		RefTargetHandle Clone(RemapDir& remap = NoRemap());

// ---- Inherited virtual methods from BaseObject ----
		// This is the name that appears in the history list (modifier stack).
		TCHAR *GetObjectName() { return MODIFIER_NAME; }
		CreateMouseCallBack* GetCreateMouseCallBack() { return NULL; } 
	};

// Initialize the class variables.
HWND SpherifyMod::hSpherifyParams = NULL;
ISpinnerControl* SpherifyMod::percentSpin = NULL;
IObjParam *SpherifyMod::iObjParams = NULL;

SpherifyMod::SpherifyMod()
	{
	ParamBlockDesc desc[] = {{ TYPE_FLOAT, NULL, TRUE }};
	MakeRefByID( FOREVER, PBLOCK_REF_INDEX, CreateParameterBlock( desc, 1 ) );
	pblock->SetValue(PERCENT_PB_INDEX, TimeValue(0), 100.0f);
	SelfNode = NULL;
	}

void SpherifyMod::SetPercent(TimeValue t, float f )
	{
	pblock->SetValue(PERCENT_PB_INDEX, t, f );	
	NotifyDependents(FOREVER, PART_GEOM, REFMSG_CHANGE);
	}

float SpherifyMod::GetPercent(TimeValue t, Interval& valid )
	{
	float f;
	pblock->GetValue(PERCENT_PB_INDEX, t, f, valid );
	return f;
	}

// This class is our deformer.  It provides some variables to store 
// data associated with the deformation, and a single method, Map, 
// which is called to deform each point.
class SpherifyDeformer : public Deformer {
	public:
	SpherifyMod *mod;
	SpherifyDeformer();
	SpherifyDeformer(SpherifyMod *m) {mod = m;}
	Point3 Map(int i, Point3 p);
	};

SpherifyDeformer::SpherifyDeformer()
	{
	};



// This is the method which deforms a single point.
Point3 SpherifyDeformer::Map(int i, Point3 p)
	{

	p = p * mod->BackMatrices[i];
	p = p * mod->CurrentMatrix;

	return p;

	}


/*===========================================================================*\
 | Modification stuff...
\*===========================================================================*/
// This method, called by the system, modifies the item
void SpherifyMod::ModifyObject(TimeValue t, ModContext &mc, ObjectState * os, INode *node ) 
	{	

	int nv = os->obj->NumPoints();

	if (t==0)
		{
		float TimeDelay,Dist;
		TimeDelay = 160.0f *16.0f;
		Dist = 100.0f;
		BackTimes.ZeroCount();
		for (int i=0;i<nv;i++)
			{
			TimeValue temp;

			temp = (TimeValue)((Length(os->obj->GetPoint(i))*TimeDelay)/Dist);
			BackTimes.Append(1,&temp,1);
			}
		}

	BackMatrices.ZeroCount();
	ModContextList mcList;
	INodeTab nodes;

	if ((iObjParams != NULL) && (SelfNode == NULL))
		{

		iObjParams->GetModContexts(mcList,nodes);
		assert(nodes.Count());

		if (SelfNode == NULL)
			ReplaceReference(SELF_NODE,nodes[0]);
		}

	CurrentMatrix = Inverse(SelfNode->GetObjectTM(t));


	for (int i=0;i<nv;i++)
		{
		Matrix3 Temp;
		Temp = SelfNode->GetObjectTM(iObjParams->GetTime()-BackTimes[i]);
		BackMatrices.Append(1,&Temp,1);
		}

	SpherifyDeformer deformer(this);

	os->obj->Deform(&deformer, TRUE);

	// This informs the system that the object may need to be re-evaluated
	// if the user moves to a new time.  We pass the channel we have 
	// modified, and the interval of our modification.
	os->obj->UpdateValidity(GEOM_CHAN_NUM, SphValidity(t));

	// Update our user interface parameter
	UpdateUI(t);
	}


Interval SpherifyMod::SphValidity(TimeValue t)
	{
	// Start our interval at forever...
	Interval valid = FOREVER;

	// Intersect with the percentage parameter to narrow it down.
	GetPercent(t, valid);

	valid.Set(t,t);

	// Return the final valdity interval.
	return valid;
	}

RefTargetHandle SpherifyMod::Clone(RemapDir& remap) 
	{
	// Make a new SpherifyMod.
	SpherifyMod* newmod = new SpherifyMod();	

	// This routine is used when cloning reference makers, to delete old 
	// reference and make a new one.
	newmod->ReplaceReference(PBLOCK_REF_INDEX, pblock->Clone(remap));

	// Return the cloned modifier.
	return(newmod);
	}

// This updates the parameters in the command panel, so that if the user 
// changes the current time and the parameters are animated, the user
// interface controls will change.
void SpherifyMod::UpdateUI(TimeValue t)
	{
	// Retrive the modifier that is being altered in the command panel.
	// A pointer to it is stored in the rollup page window data.
	SpherifyMod *sm = (SpherifyMod *)GetWindowLong( hSpherifyParams, 
		GWL_USERDATA );

	// There may be many modifiers in the scene.  Make sure 
	// this is the one we are editing...
	if ( sm == this ) {
		percentSpin->SetValue( GetPercent(t), FALSE );
		}
	}

// This method is inherited from the Animatable class.
// It is called by the system when the user is in the position of being able
// to edit the modifiers parameters in the command panel.  
void SpherifyMod::BeginEditParams( IObjParam *ip, ULONG flags, Animatable *prev )
	{
	iObjParams = ip;
	hSpherifyParams = ip->AddRollupPage( 
		hInstance, 
		MAKEINTRESOURCE(IDD_SPHERIFY),
		SpherifyDialogProc,
		PARAMETERS_NAME, 
		(LPARAM)this );		
		
	ip->RegisterDlgWnd( hSpherifyParams );

	// This updates the parameters in the UI to make sure they are current.
 	UpdateUI( ip->GetTime() );
	}

// This method is inherited from the Animatable class.
// It is called by the system when we need to terminate the editing of our
// parameters in the command panel.  
void SpherifyMod::EndEditParams( IObjParam *ip, ULONG flags, Animatable *next )
	{
	if ( flags&END_EDIT_REMOVEUI ) {		
		// Unregister the parameters window...
		ip->UnRegisterDlgWnd( hSpherifyParams );		
		// Delete the window
		ip->DeleteRollupPage( hSpherifyParams );		
		// The interface pointer is only valid within BeginEditParams and 
		// EndEditParams.  Set it to null for safety.
		hSpherifyParams = NULL;				
	} 
	else {
		SetWindowLong( hSpherifyParams, GWL_USERDATA, 0 );
	}
}

// This is the dialog proc for the parameters rollup page.
BOOL CALLBACK SpherifyDialogProc( HWND hDlg, UINT message, WPARAM wParam, 
	LPARAM lParam )
	{
	SpherifyMod *sm = (SpherifyMod *)GetWindowLong( hDlg, GWL_USERDATA );

	if ( !sm && message != WM_INITDIALOG ) return FALSE;
	
	switch ( message ) {
		case WM_INITDIALOG:
			// Retrieve a pointer to the modifier we are editing.
		 	sm = (SpherifyMod *)lParam;
			// Store this pointer into the window data for the rollup page.
		 	SetWindowLong( hDlg, GWL_USERDATA, (LONG)sm );
		 	// Init the spinner
		 	sm->percentSpin = GetISpinner(GetDlgItem(hDlg,IDC_PERCENT_SPIN));
			sm->percentSpin->SetLimits( 0.0f, 100.0f, TRUE );
			sm->percentSpin->LinkToEdit( GetDlgItem(hDlg,IDC_PERCENT), 
				EDITTYPE_FLOAT );
		 	return TRUE;

		case WM_DESTROY:
			ReleaseISpinner( sm->percentSpin );			
			return FALSE;

		case WM_MOUSEACTIVATE:
			sm->iObjParams->RealizeParamPanel();
			return FALSE;

		case WM_LBUTTONDOWN: case WM_LBUTTONUP:	case WM_MOUSEMOVE:   			
   			sm->iObjParams->RollupMouseMessage(hDlg, message, wParam, lParam);
			return FALSE;

		case CC_SPINNER_CHANGE:
			switch ( LOWORD(wParam) ) {
				case IDC_PERCENT_SPIN:
					// First we update the modifiers parameters
					sm->SetPercent( 
						sm->iObjParams->GetTime(), 
						sm->percentSpin->GetFVal() );	
					// Then we redraw the viewports.
					// Note that we specify REDRAW_INTERACTIVE. This means that
					// we are doing an interactive adjustment so the rendering
					// level may degrade to maintain interactivity.
					sm->iObjParams->RedrawViews(
						sm->iObjParams->GetTime(),REDRAW_INTERACTIVE);
					break;
				}
		case CC_SPINNER_BUTTONDOWN:
			// Begin holding for Undo
			theHold.Begin(); 
			break;

		case CC_SPINNER_BUTTONUP:
			// When the user lets up on the button after adjusting a spinner,
			// this message is sent. We call redraw viewports one more time
			// specifying REDRAW_END to undegrade any degradation.
			// This message is also sent when the user presses ENTER after
			// typing in an edit field.
			sm->iObjParams->RedrawViews(sm->iObjParams->GetTime(),REDRAW_END);
			break;

		default:
			break;
		}
	return FALSE;
	}

// Inherited from ReferenceMaker
// This is only called if the we MAKE references to other things.
// Our modifier makes a reference to its parameter block.
RefResult SpherifyMod::NotifyRefChanged( 
		Interval changeInt,
		RefTargetHandle hTarget, 
		PartID& partID, 
		RefMessage message )
	{
	switch (message) {
		case REFMSG_GET_PARAM_DIM: { 
			GetParamDim *gpd = (GetParamDim*)partID;
			switch (gpd->index) {
				case PERCENT_PB_INDEX:
					gpd->dim = defaultDim;
					break;
				}
			return REF_STOP; 
			}

		case REFMSG_GET_PARAM_NAME: {
			GetParamName *gpn = (GetParamName*)partID;
			switch (gpn->index) {
				case PERCENT_PB_INDEX:
					gpn->name = PERCENT_PARAM_NAME;
					break;
				}
			return REF_STOP; 
			}
		case REFMSG_TARGET_DELETED: {
			if (hTarget==pblock) 
					pblock = NULL;									
			else if (hTarget == SelfNode)
				  SelfNode = NULL;
			break;
			}

		}
	return REF_SUCCEED;
	}

/*===========================================================================*\
 | Class Descriptor
\*===========================================================================*/
class SpherifyClassDesc : public ClassDesc {
	public:
	int 			IsPublic() { return 1; }
	void * 			Create(BOOL loading=FALSE) { return new SpherifyMod(); }
	const TCHAR * 	ClassName() { return SPHERIFY_CNAME; }
	SClass_ID 		SuperClassID() { return OSM_CLASS_ID; }
	Class_ID 		ClassID() { return Class_ID(SPH_CLASS_ID1,SPH_CLASS_ID2); }
	const TCHAR* 	Category() { return CATEGORY_NONE;  }
	};

static SpherifyClassDesc SpherifyCD;

/*===========================================================================*\
 | DLL Functions
\*===========================================================================*/
// This function is called by Windows when the DLL is loaded.  This 
// function may also be called many times during time critical operations
// like rendering.  Therefore developers need to be careful what they
// do inside this function.  In the code below, note how after the DLL is
// loaded the first time only a few statements are executed.
int controlsInit = FALSE;
BOOL WINAPI DllMain(HINSTANCE hinstDLL,ULONG fdwReason,LPVOID lpvReserved) 
	{	
	// Hang on to this DLL's instance handle.
	hInstance = hinstDLL;

	if (! controlsInit) {
		controlsInit = TRUE;
		
		// Initialize MAX's custom controls
		InitCustomControls(hInstance);
		
		// Initialize Win95 controls
		InitCommonControls();
	}
	
	return(TRUE);
	}

// This function returns the number of plug-in classes this DLL implements
__declspec( dllexport ) int 
LibNumberClasses() { return 1; }

// This function return the ith class descriptor. We have one.
__declspec( dllexport ) ClassDesc* 
LibClassDesc(int i) { return &SpherifyCD; }

// This function returns a string presented if the DLL is missing or
// from the Summary dialog box.
__declspec( dllexport ) const TCHAR *
LibDescription() {
	return _T("Spherify Modifier"); 
	}

// This function returns a pre-defined constant indicating the version of 
// the system under which it was compiled.  It is used to allow the system
// to catch obsolete DLLs.
__declspec( dllexport ) ULONG 
LibVersion() { return VERSION_3DSMAX; }
